/** Copyright (c) 2023. All rights reserved.
  *
  * This software and all trademarks, trade names, and logos included herein are
  * the property of Digital.ai Software, Inc. and its affiliates, subsidiaries,
  * and licensors.
  */
package com.xebialabs.xlrelease.plugin.platform.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.{
  BaseConfiguration,
  Configuration,
  Release
}
import com.xebialabs.xlrelease.plugin.platform.connector.HttpClientWrapper._
import com.xebialabs.xlrelease.plugin.platform.converter.BaseConfigurationConverter._
import com.xebialabs.xlrelease.plugin.platform.domain.PlatformServer
import com.xebialabs.xlrelease.plugin.platform.utils.StringUtils._
import com.xebialabs.xlrelease.repository.IdMatchers.{FolderId, ReleaseId}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.script.EncryptionHelper
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.service.{
  ConfigurationVariableService,
  SharedConfigurationService
}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import spray.json._

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Service
class ReleaseEventService extends Logging {

  @Autowired
  var sharedConfigurationService: SharedConfigurationService = _
  @Autowired
  var configurationVariableService: ConfigurationVariableService = _

  def processRelease(message: String, release: Release): Unit = {
    logger.debug(message)
    val platformContext = getPlatformContext(release)
    if (!platformContext.equalsIgnoreCase(ReleaseEventsService.NOT_AVAILABLE)) {
      logger.debug(s"Release [${release.getId} contains platform variable]")
      val payload = buildPayload(release, platformContext)
      val platform = getPlatformServer(release.getId)
      if (platform != null) {
        // make sure secrets are decrypted in a listener after payload is already built
        EncryptionHelper.decrypt(platform)
        val resp = platform.getClient.getWrapper
          .post(ReleaseEventsService.URL_API_PATH, payload.toString())
          .doRequest[String]()
          .flatMap(
            _.checkResponse(
              "Error occurred while connecting Delivery Insights Service"
            )
          )

        logger.debug(
          s"\n${message}\n " +
            s"Platform response: \n " +
            s"RESPONSE_STATUS = ${resp.get.getStatusCode} \n " +
            s"RESPONSE_BODY = \n ${resp.get.getBody} \n"
        )
      } else {
        logger.error("Did not find any configured Delivery Insights Service")
      }
    } else {
      logger.debug(
        s"Release [${release.getId} does not contain platform_context variable.] Skipping ..."
      )
    }

  }

  def processSharedConfiguration(
      conf: BaseConfiguration,
      overwrite: Boolean = false
  ): Unit = {
    val folderId = conf.getFolderId.orElse(Ids.ROOT_FOLDER_ID)
    val platform = getPlatformServer(folderId)
    if (platform != null) {
      val platformServer = platform.asInstanceOf[PlatformServer]
      val syncList =
        platformServer.getProperty[java.util.List[String]]("sync_list").asScala
      if (syncList.contains(conf.getType.toString)) {
        logger.info(
          s"Found configuration [${conf.getId} - ${conf.getTitle} (${conf.getType})] to be synced with platform at [${platformServer.getUrl}]"
        )
        syncConf(conf.asInstanceOf[Configuration], overwrite, platformServer)
      } else if (conf.getType.equals(platformServer.getType)) {
        // if we are creating PlatformServer, sync what we had already
        syncList.foreach { sharedType =>
          getSharedConfigurations(folderId, sharedType).foreach { shared =>
            syncConf(shared, overwrite, platformServer)
          }
        }
      }
    } else {
      logger.warn("Did not find any configured Delivery Insights Service")
    }

  }

  private def syncConf(
      conf: Configuration,
      overwrite: Boolean,
      platformServer: PlatformServer
  ): Unit = {
    configurationVariableService.resolve(conf)
    toPluginConfigurationRequest(conf, overwrite).map { req =>
      Try(platformServer.getClient.configurePlugin(req)) match {
        case Failure(exception) =>
          logger.error(
            s"Failed to store on Platform [${conf.getTitle}]",
            exception
          )
        case Success(_) =>
          logger.info(s"Stored on Platform [${conf.getTitle}]")
      }
    }
  }

  def buildPayload(release: Release, platformContext: String): JsValue = {
    val releaseJson = CiSerializerHelper.serializeWithMaskedPasswords(release)
    // remove $ from elements in json
    val initialPayload =
      buildInitialPayload(platformContext, releaseJson.replaceAll("\\$", ""))
    initialPayload.parseJson
  }

  private def buildInitialPayload(
      platform_context: String,
      releaseJson: String
  ) = {
    val quote = "\""
    val initialPayload: String = {
      s"{${quote}platform_context${quote}: ${platform_context}, ${quote}xlrelease_details${quote} : ${releaseJson}}"
    }
    initialPayload
  }

  def getPlatformServer(folderOrTemplateId: String): PlatformServer = {
    val optionalFolderId = folderOrTemplateId match {
      case FolderId(id)                 => Some(id)
      case ReleaseId(id)                => Some(Ids.findFolderId(id))
      case rootId if Ids.isRoot(rootId) => Some(Ids.ROOT_FOLDER_ID)
      case _                            => None
    }
    optionalFolderId
      .flatMap { folderId =>
        val platformConfigType = Type.valueOf(classOf[PlatformServer])
        val platformConfigurations = sharedConfigurationService
          .searchByTypeAndTitle(platformConfigType, null, folderId, false)
          .asScala
        platformConfigurations
          .sortBy(directoryDepth)(Ordering.Int.reverse)
          .headOption
      }
      .orNull
      .asInstanceOf[PlatformServer]
  }

  def getSharedConfigurations(
      folderOrTemplateId: String,
      typeConf: String
  ): Seq[Configuration] = {
    val optionalFolderId = folderOrTemplateId match {
      case FolderId(id)                 => Some(id)
      case ReleaseId(id)                => Some(Ids.findFolderId(id))
      case rootId if Ids.isRoot(rootId) => Some(Ids.ROOT_FOLDER_ID)
      case _                            => None
    }
    optionalFolderId
      .map { folderId =>
        val configType = Type.valueOf(typeConf)
        try {
          sharedConfigurationService
            .searchByTypeAndTitle(configType, null, folderId, false)
            .asScala
            .toSeq
        } catch {
          case e: Exception =>
            logger.warn(
              s"Error searching type [$typeConf] on folder [$folderId]: ${e.getMessage}"
            )
            Seq.empty
        }
      }
      .getOrElse(Seq.empty)
  }

  def directoryDepth(conf: Configuration): Int = {
    if (conf.getFolderId == null) 0 else conf.getFolderId.count(_ == '/')
  }

  def getPlatformContext(release: Release): String = {
    val platform_context =
      getReleaseVariable(release, ReleaseEventsService.CONTINUUM_CONTEXT)
    platform_context
  }

  def getReleaseVariable(release: Release, platformContext: String): String = {
    val releaseVariable: String = {
      release.getVariables.asScala
        .find(_.getKey == platformContext)
        .map(_.getValueAsString)
        .getOrElse(ReleaseEventsService.NOT_AVAILABLE)
    }
    releaseVariable
  }
}

object ReleaseEventsService {
  val CONTINUUM_CONTEXT = "platform_context"
  val URL_API_PATH = "api/xlrelease_status"
  val NOT_AVAILABLE = "platformContext"
}
