/** Copyright (c) 2023. All rights reserved.
 *
 * This software and all trademarks, trade names, and logos included herein are
 * the property of Digital.ai Software, Inc. and its affiliates, subsidiaries,
 * and licensors.
 */
package com.xebialabs.xlrelease.plugin.platform

import com.fasterxml.jackson.annotation.{JsonCreator, JsonInclude}
import com.fasterxml.jackson.databind.annotation.JsonDeserialize
import com.fasterxml.jackson.databind.deser.std.UntypedObjectDeserializer
import com.fasterxml.jackson.databind.{DeserializationFeature, ObjectMapper, SerializationFeature}
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty
import com.xebialabs.deployit.checks.Checks.MissingArgumentException
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.plugin.platform.connector.{ConnectionConfiguration, HttpClientWrapper, ProxyConfiguration, ServerConfiguration}
import com.xebialabs.xlrelease.plugin.platform.domain.{PlatformPythonScript, PlatformServer, RemovalReason}
import feign._
import feign.hc5.ApacheHttp5Client
import feign.jackson.{JacksonDecoder, JacksonEncoder}
import feign.slf4j.Slf4jLogger
import grizzled.slf4j.Logging
import org.apache.commons.io.IOUtils
import org.apache.http.HttpHeaders

import java.io.Closeable
import java.lang.reflect.Type
import java.nio.charset.StandardCharsets
import scala.annotation.meta.field
import scala.beans.BeanProperty
import scala.util.{Failure, Success, Try}

// scalastyle:off
object Client {
  def apply(platformServer: PlatformServer): Client = {
    if (platformServer == null)
      throw new MissingArgumentException(
        "Delivery Insights Service is not defined"
      )
    val apiToken = platformServer.getApiToken
    create(platformServer, apiToken)
  }

  def apply(platformPythonScript: PlatformPythonScript): Client = {
    val taskApiToken = platformPythonScript.getApiToken
    val platformServer = platformPythonScript.getPlatformServer
    if (platformServer == null)
      throw new MissingArgumentException(
        "Delivery Insights Service is not defined"
      )
    val apiToken =
      if (isNotEmpty(taskApiToken)) taskApiToken else platformServer.getApiToken
    create(platformServer, apiToken)
  }

  private def create(
                      platformServer: PlatformServer,
                      apiToken: String
                    ): Client = {
    val proxyPass = platformServer.getProxyPassword
    val configuration = ServerConfiguration(
      platformServer.getUrl,
      platformServer.getUsername,
      platformServer.getPassword,
      PasswordEncrypter.getInstance().ensureDecrypted(apiToken),
      ProxyConfiguration(
        platformServer.getProxyHost,
        platformServer.getProxyPort,
        platformServer.getProxyUsername,
        proxyPass,
        platformServer.getDomain
      ),
      ConnectionConfiguration(
        platformServer.getAuthenticationMethod,
        platformServer.getTimeOutSeconds
      )
    )
    new Client(configuration)
  }
}

case class AddApplicationPluginRequest(
                                        @BeanProperty id: String,
                                        @(JsonDeserialize @field)(contentUsing = classOf[UntypedObjectDeserializer])
                                        @BeanProperty servers: ListMapResponseType
                                      ) {
  @JsonCreator() def this() = this("", null)
}
@JsonDeserialize(builder = classOf[UntypedObjectDeserializer])
case class GenericResponse()

@JsonDeserialize(builder = classOf[UntypedObjectDeserializer])
case class GenericPlatformResponse()

case class GenericListResponse(
                                @BeanProperty pagination: Pagination,
                                @BeanProperty pagination_headers: PaginationHeaders,
                                @(JsonDeserialize @field)(contentUsing = classOf[UntypedObjectDeserializer])
                                @BeanProperty results: ListMapResponseType
                              ) extends PlatformPagedResponse[ListMapResponseType]

case class IdResponse(
                       @BeanProperty created_by: String,
                       @BeanProperty ErrorDetail: String,
                       @BeanProperty ErrorMessage: String,
                       @BeanProperty Method: String,
                       @BeanProperty Response: String
                     )

case class ValidateInstanceResponse(@BeanProperty result: String)

trait LimitRequest {
  def limit: String
}

case class PaginationHeaders(
                              comment: String,
                              current_page: Int,
                              has_next: Boolean,
                              has_previous: Boolean,
                              size: Int,
                              total_elements: Int,
                              total_pages: Int
                            )

case class Pagination(
                       count: Int,
                       next_start: Int,
                       previous_start: Int,
                       start: Int,
                       total_available: Int
                     )

trait PlatformPagedResponse[T] {

  def pagination: Pagination

  def pagination_headers: PaginationHeaders

  def results: T
}

case class Phase(
                  code_complete: Boolean,
                  description: String,
                  id: String,
                  name: String
                )

case class ProgressionResponse(
                                @BeanProperty created_dt: String,
                                @BeanProperty id: String,
                                @BeanProperty name: String,
                                @BeanProperty phases: List[Phase]
                              )

trait PluginConfigurationRequest

case class JenkinsPluginConfigurationRequest(
                                              plugin: String = "jenkins",
                                              description: String,
                                              name: String,
                                              user: String,
                                              password: String = null,
                                              api_token: String,
                                              url: String,
                                              team_id: String = null,
                                              isdefault: Boolean = false,
                                              overwrite: Boolean = false
                                            ) extends PluginConfigurationRequest

case class JiraPluginConfigurationRequest(
                                           plugin: String = "jiraplugin",
                                           description: String,
                                           name: String,
                                           user: String,
                                           password: String,
                                           url: String,
                                           improve_types: String,
                                           maintain_types: String,
                                           team_id: String = null,
                                           isdefault: Boolean = false,
                                           overwrite: Boolean = false,
                                           webhook_handler_name: String = null
                                         ) extends PluginConfigurationRequest

case class GithubPluginConfigurationRequest(
                                             plugin: String = "github",
                                             name: String,
                                             description: String,
                                             owner: String,
                                             url: String,
                                             team_id: String = null,
                                             api_token: String,
                                             isdefault: Boolean = false,
                                             overwrite: Boolean = false
                                           ) extends PluginConfigurationRequest

case class GitlabPluginConfigurationRequest(
                                             plugin: String = "gitlab",
                                             name: String,
                                             description: String,
                                             url: String,
                                             api_token: String,
                                             team_id: String = null,
                                             isdefault: Boolean = false,
                                             overwrite: Boolean = false
                                           ) extends PluginConfigurationRequest

case class AgilityPluginConfigurationRequest(
                                              plugin: String = "v1plugin",
                                              description: String,
                                              name: String,
                                              url: String,
                                              api_token: String,
                                              isdefault: Boolean = false,
                                              overwrite: Boolean = false,
                                              webhook_handler_name: String = null
                                            ) extends PluginConfigurationRequest

case class BitbucketServerPluginConfigurationRequest(
                                                      plugin: String = "stash",
                                                      description: String,
                                                      name: String,
                                                      url: String,
                                                      user: String,
                                                      password: String,
                                                      isdefault: Boolean = false,
                                                      overwrite: Boolean = false
                                                    ) extends PluginConfigurationRequest

case class BitbucketCloudPluginConfigurationRequest(
                                                     plugin: String = "bitbucket",
                                                     description: String,
                                                     name: String,
                                                     url: String,
                                                     user: String,
                                                     password: String,
                                                     isdefault: Boolean = false,
                                                     overwrite: Boolean = false
                                                   ) extends PluginConfigurationRequest

case class CreatePackageRequest(
                                 name: String,
                                 team: String,
                                 description: String = null,
                                 progression: String = null
                               )

case class CreatePackageRevisionRequest(
                                         progression_id: String,
                                         phase_id: String,
                                         repository_id: String,
                                         branch: String,
                                         version: String,
                                         alias: String = null
                                       )

case class PromoteAPackageRequest(
                                   package_id: String,
                                   revision: String,
                                   from_phase_id: String,
                                   to_phase_id: String,
                                   progression_id: String = null
                                 )
case class CardQueryRequest(
                             progression_id: String = null,
                             package_id: String = null,
                             revision: String = null
                           )

case class CreateProgressionRequest(name: String)

case class AddProgressionPhasesRequest(
                                        name: String,
                                        phases: Seq[AddProgressionPhasesPhaseRequest]
                                      )

case class AddProgressionPhasesPhaseRequest(
                                             name: String,
                                             description: String,
                                             before_phase: String = null,
                                             code_complete: Boolean = false
                                           )

case class RemoveRevisionRequest(delivery_status: RemovalReason)


@Headers(
  Array(
    "Accept: application/json",
    "Content-Type: application/json"
  )
)
trait PlatformClient {

  // ===================================================================
  //
  // Continuum Requests, to be removed in a follow-up story
  //
  // Some of these are used -- directly or indirectly --  in
  //   - plugin/platform/service/ReleaseEventService.scala
  //   - plugin/platform/listener/ReleaseEventsListener.scala
  //
  // ===================================================================

  @RequestLine("POST /api/configure_plugin_instance")
  def configureJenkinsPlugin(
                              plugin: JenkinsPluginConfigurationRequest
                            ): IdResponse

  @RequestLine("POST /api/configure_plugin_instance")
  def configureJiraPlugin(plugin: JiraPluginConfigurationRequest): IdResponse

  @RequestLine("POST /api/configure_plugin_instance")
  def configureGithubPlugin(
                             plugin: GithubPluginConfigurationRequest
                           ): IdResponse

  @RequestLine("POST /api/configure_plugin_instance")
  def configureGitlabPlugin(
                             plugin: GitlabPluginConfigurationRequest
                           ): IdResponse

  @RequestLine("POST /api/configure_plugin_instance")
  def configureAgilityPlugin(
                              plugin: AgilityPluginConfigurationRequest
                            ): IdResponse

  @RequestLine("POST /api/configure_plugin_instance")
  def configureBitbucketServerPlugin(
                                      plugin: BitbucketServerPluginConfigurationRequest
                                    ): IdResponse

  @RequestLine("POST /api/configure_plugin_instance")
  def configureBitbucketCloudPlugin(
                                     plugin: BitbucketCloudPluginConfigurationRequest
                                   ): IdResponse

  @RequestLine("POST /api/create_package")
  def createPackage(request: CreatePackageRequest): GenericResponse

  @RequestLine("POST /api/add_progression_phases")
  def addProgressionPhases(
                            request: AddProgressionPhasesRequest
                          ): GenericResponse

  // ===================================================================
  //
  // Platform Requests
  //
  // ===================================================================

  @RequestLine("GET /insight/v1/validate_instance/")
  def checkConnection(): ValidateInstanceResponse

  @RequestLine("GET /insight/v1/applications/?start={start}&count={count}")
  def listApplications(@Param("start") start: Int, @Param("count") count: Int): GenericListResponse

  @RequestLine("GET /insight/v1/applications/{id}")
  def getApplication(@Param("id") id: String): MapResponseType

  @RequestLine("GET /insight/v1/configuration/items/{id}")
  def getConfiguration(@Param("id") id: String): MapResponseType

  @RequestLine(
    "GET /insight/v1/cards/?progression_id={progression_id}&package_id={package_id}&revision={revision}"
  )
  def listCards(
                 @Param("progression_id") progression_id: String,
                 @Param("package_id") package_id: String,
                 @Param("revision") revision: String
               ): GenericListResponse

  @RequestLine("POST /insight/v1/cards/{card_id}/revisions/{revision_number}/deliveries/")
  def removeRevision(
                       @Param("card_id") card_id: String,
                       @Param("revision_number") revision_number: String,
                       request: RemoveRevisionRequest
                     ): Unit

  @RequestLine("POST /insight/v1/packages/{package_id}/revisions/")
  def createPackageRevision(
                             @Param("package_id") package_id: String,
                             request: CreatePackageRevisionRequest
                           ): MapResponseType

  @RequestLine("POST /insight/v1/promotions/")
  def promoteAPackage(request: PromoteAPackageRequest): Unit

  @RequestLine("GET /insight/v1/progressions/?start={start}&count={count}")
  def listProgressions(@Param("start") start: Int, @Param("count") count: Int): GenericListResponse

  @RequestLine("GET /insight/v1/progressions/{id}")
  def getProgression(@Param("id") id: String): MapResponseType

  @RequestLine("GET /insight/v1/progressions/?package_id={id}")
  def listProgressionsByPackageId(@Param("id") id: String): GenericListResponse

  @RequestLine("GET /insight/v1/packages/{id}")
  def getPackage(@Param("id") id: String): MapResponseType

  @RequestLine("POST /insight/v1/progressions/")
  def createProgression(request: CreateProgressionRequest): ProgressionResponse

}

class TokenAuthRequestInterceptor(token: String) extends RequestInterceptor {
  override def apply(template: RequestTemplate): Unit = {
    if (isNotEmpty(token)) {
      template.header(HttpHeaders.AUTHORIZATION, s"Token $token")
    }
  }
}

class Client(configuration: ServerConfiguration)
  extends Logging
    with AutoCloseable
    with Closeable {

  private val cw: HttpClientWrapper =
    HttpClientWrapper(configuration).setUriEncoded(true)

  private var _closed: Boolean = false

  def getWrapper: HttpClientWrapper = cw

  private val client = HttpClientWrapper.getDefaultClient(configuration)

  private val encoderMapper: ObjectMapper = new ObjectMapper()
    .setSerializationInclusion(JsonInclude.Include.NON_NULL)
    .configure(SerializationFeature.INDENT_OUTPUT, true)
    .registerModule(DefaultScalaModule)

  private val jacksonEncoder = new JacksonEncoder(encoderMapper)

  private val decoderMapper: ObjectMapper = new ObjectMapper()
    .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
    .registerModule(DefaultScalaModule)

  private val jacksonDecoder = new JacksonDecoder(decoderMapper)

  val responseMapper: ResponseMapper = (response: Response, respType: Type) => {
    val status = response.status()
    val responseTxt =
      IOUtils.toString(response.body().asReader(StandardCharsets.UTF_8))
    if (status == 280) {
      val url = response.request().url()
      if (url.contains("remove") || url.contains("delete")) {
        throw EntryNotFound(status, responseTxt)
      } else if (url.contains("create")) {
        throw DuplicateEntry(status, responseTxt)
      } else {
        throw OperationFailed(status, responseTxt)
      }
    } else {
      response.toBuilder.body(responseTxt, StandardCharsets.UTF_8).build()
    }
  }

  private val platformClient = Feign
    .builder()
    .encoder(jacksonEncoder)
    .mapAndDecode(responseMapper, jacksonDecoder)
    .client(new ApacheHttp5Client(client))
    .logger(new Slf4jLogger(classOf[Client]))
    .logLevel(Logger.Level.FULL)
    .requestInterceptor(new TokenAuthRequestInterceptor(configuration.apiToken))
    .target(classOf[PlatformClient], configuration.url)

  def checkConnection(): Boolean = {
    invoke {
      val response = platformClient.checkConnection()
      val result = response.result
      result == "success"
    }
  }

  // ===================================================================
  //
  // Continuum Requests, to be removed in a follow-up story
  //
  // Some of these are used -- directly or indirectly --  in
  //   - plugin/platform/service/ReleaseEventService.scala
  //   - plugin/platform/listener/ReleaseEventsListener.scala
  //
  // ===================================================================

  def configurePlugin(
                       pluginConfiguration: PluginConfigurationRequest
                     ): AnyRef = {
    invoke {
      pluginConfiguration match {
        case p: JiraPluginConfigurationRequest =>
          // TODO dirty fix, new field is added in CTM and is not backwards compatible, to think something better
          var copy: JiraPluginConfigurationRequest = p
          platformClient.configureJiraPlugin(copy)
        case p: JenkinsPluginConfigurationRequest =>
          platformClient.configureJenkinsPlugin(p)
        case p: GithubPluginConfigurationRequest =>
          platformClient.configureGithubPlugin(p)
        case p: GitlabPluginConfigurationRequest =>
          platformClient.configureGitlabPlugin(p)
        case p: BitbucketCloudPluginConfigurationRequest =>
          platformClient.configureBitbucketCloudPlugin(p)
        case p: BitbucketServerPluginConfigurationRequest =>
          platformClient.configureBitbucketServerPlugin(p)
        case p: AgilityPluginConfigurationRequest =>
          // TODO dirty fix, new field is added in CTM and is not backwards compatible, to think something better
          var copy: AgilityPluginConfigurationRequest = p
          platformClient.configureAgilityPlugin(copy)
      }
    }
  }

  def createPackage(request: CreatePackageRequest): GenericResponse = {
    invoke {
      platformClient.createPackage(request)
    }
  }

  def addProgressionPhases(
                            request: AddProgressionPhasesRequest
                          ): GenericResponse = {
    invoke {
      platformClient.addProgressionPhases(request)
    }
  }

  // ===================================================================
  //
  // Platform Requests
  //
  // ===================================================================

  def promoteAPackage(request: PromoteAPackageRequest): Unit = {
    invoke {
      platformClient.promoteAPackage(request)
    }
  }

  def createPackageRevision(
                             package_id: String,
                             request: CreatePackageRevisionRequest
                           ): MapResponseType = {
    invoke {
      platformClient.createPackageRevision(package_id, request)
    }
  }

  def listApplications(start: Int, count: Int): GenericListResponse = {
    invoke {
      platformClient.listApplications(start, count)
    }
  }
  def listCards(
                 progression_id: String,
                 package_id: String,
                 revision: String
               ): GenericListResponse = {
    invoke {
      platformClient.listCards(progression_id, package_id, revision)
    }
  }

  def removeRevision(card_id: String, revision_number: String, request: RemoveRevisionRequest): Unit = {
    invoke {
      platformClient.removeRevision(card_id, revision_number, request)
    }
  }

  def getApplication(application_id: String): MapResponseType = {
    invoke {
      platformClient.getApplication(application_id)
    }
  }

  def getConfiguration(configuration_id: String): MapResponseType = {
    invoke {
      platformClient.getConfiguration(configuration_id)
    }
  }

  def listProgressions(start: Int, count: Int): GenericListResponse = {
    invoke {
      platformClient.listProgressions(start, count)
    }
  }

  def getProgression(progression_id: String): MapResponseType = {
    invoke {
      platformClient.getProgression(progression_id)
    }
  }

  def listProgressionsByPackageId(packageId: String): GenericListResponse = {
    invoke {
      platformClient.listProgressionsByPackageId(packageId)
    }
  }

  def createProgression(
                         request: CreateProgressionRequest
                       ): ProgressionResponse = {
    invoke {
      platformClient.createProgression(request)
    }
  }

  def closed(): Boolean = this._closed

  override def close(): Unit = {
    client.close()
    _closed = true
  }

  private def invoke[T](body: => T): T = {
    if (this._closed) {
      throw new IllegalStateException(
        "Connection was already closed, can't reuse."
      )
    }

    Try { // TODO error handling, check when connections are closed, etc...
      body
    } match {
      case Success(response) =>
        response
      case Failure(ex: PlatformException) =>
        throw ex
      case Failure(ex: FeignException) =>
        val msg = ex.getMessage
        logger.error(msg, ex)
        throw new IllegalStateException(msg, ex)
      case Failure(ex) =>
        val msg = "Encountered error"
        logger.warn(msg, ex)
        throw new IllegalStateException(msg, ex)
    }
  }

}

trait PlatformException

case class DuplicateEntry(override val status: Int, msg: String)
  extends FeignException(status, msg)
    with PlatformException

case class EntryNotFound(override val status: Int, msg: String)
  extends FeignException(status, msg)
    with PlatformException

case class OperationFailed(override val status: Int, msg: String)
  extends FeignException(status, msg)
    with PlatformException
