#
# Copyright (c) 2019. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

"""
List, create, destroy and refresh Delphix timeflows
"""
# TODO:
#    implement debug flag

from delphixpy.web import database
from delphixpy.web import snapshot
from delphixpy.web.vo import TimeflowPointLocation
from delphixpy.web.vo import TimeflowPointTimestamp
from delphixpy.web.vo import TimeflowPointSemantic

VERSION = 'v.0.2.003'

class DxTimeflow(object):
    """Shared methods for timeflows """

    def __init__(self, engine):
        self.engine = engine


    def get_timeflow_reference(self, db_name):
        """
        Return current_timeflow for the db_name

        db_name: The database name to retrieve current_timeflow
        """

        db_lst = database.get_all(self.engine)

        for db_obj in db_lst:
            if db_obj.name == db_name:
                return db_obj.current_timeflow

        raise Exception('Timeflow reference not found for {}'.format(
            db_name))


    def find_snapshot(self, database_ref, timestamp, snap_name=None,
                      snap_time=None):
        """
        Method to find a snapshot by name

        database_obj: database reference for the snapshot lookup
        snap_name: name of the snapshot. Default: None
        snap_time: time of the snapshot. Default: None
        """

        print("Finding snapshot with timestamp %s\n," % str(timestamp))
        snapshots = snapshot.get_all(self.engine, database=database_ref)
        matches = []
        for snapshot_obj in snapshots:
            if (str(snapshot_obj.name).startswith(timestamp) and
               snap_name is not None):

                matches.append(snapshot_obj)

            elif (str(snapshot_obj.latest_change_point.timestamp).startswith(timestamp)
                  and snap_time is not None):

                matches.append(snapshot_obj)

        if len(matches) == 1:
            return matches[0]

        elif len(matches) > 1:
            raise Exception('{}: The name specified was not specific '
                                'enough. More than one match found.\n'.format(
                                self.engine.address))

        elif len(matches) < 1:
            raise Exception('{}: No matches found for the time '
                                'specified.\n'.format(self.engine.address))


    def set_timeflow_point(self, container_obj,
                           timestamp='LATEST'):
        """
        This method returns the reference of the timestamp specified.
        container_obj: Delphix object containing the snapshot/timeflow to be
                       provisioned.
        timestamp_type: Type of timestamp - SNAPSHOT or TIME
        timestamp: Name of timestamp/snapshot. Default: Latest
        """
        if timestamp.upper() == "LATEST":
            timeflow_point_parameters = TimeflowPointSemantic()
            timeflow_point_parameters.container = container_obj.reference
            timeflow_point_parameters.location = "LATEST_SNAPSHOT"

        elif timestamp.startswith("@"):
            snapshot_obj = self.find_snapshot(container_obj.reference,
                                              timestamp, snap_name=True)

            if snapshot_obj:
                timeflow_point_parameters=TimeflowPointLocation()
                timeflow_point_parameters.timeflow = snapshot_obj.timeflow
                timeflow_point_parameters.location = \
                               snapshot_obj.latest_change_point.location

            else:
                raise Exception('ERROR: Was unable to use the '
                                    'specified snapshot {}'
                                    'for database {}'.format(timestamp,
                                    container_obj.name))

        elif timestamp:
            snapshot_obj = self.find_snapshot(container_obj.reference,
                                              timestamp, snap_time=True)

            if snapshot_obj:
                timeflow_point_parameters=TimeflowPointTimestamp()
                timeflow_point_parameters.timeflow = snapshot_obj.timeflow
                timeflow_point_parameters.timestamp = \
                               snapshot_obj.latest_change_point.timestamp

            elif snapshot_obj is None:
                print('Was unable to find a suitable time'
                                '  for {} for database {}'.format(
                                (timestamp, container_obj.name)))
        timeflow_point_parameters.container = container_obj.reference
        return timeflow_point_parameters