#
# Copyright (c) 2019. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

"""
Module that provides lookups of references and names of Delphix objects.
"""

from delphixpy.exceptions import RequestError
from delphixpy.exceptions import HttpError
from delphixpy.exceptions import JobError
from delphixpy.web import repository
from delphixpy.web import source
from delphixpy.web.selfservice import bookmark
from delphixpy.web.selfservice import branch
from delphixpy.web.selfservice import template
from delphixpy.web.selfservice import container


VERSION = 'v.0.2.0019'


def find_template_ref(engine, template_name):
    templates = template.get_all(engine)
    for t in templates:
        if t.name == template_name:
            return(t.reference)

    raise Exception('{} was not found.\n'.format(template_name))

def find_container_ref(engine, template_name, container_name):
    template_ref = find_template_ref(engine, template_name)

    containers = container.get_all(engine)
    for c in containers:
        if c.template != template_ref:
            continue

        if c.name == container_name:
            return(c.reference)

    raise Exception('{} was not found in template {}.\n'.format(container_name, template_name))

def find_branch_ref(engine, template_name, container_name, branch_name):
    container_ref = find_container_ref(engine, template_name, container_name)
    branches = branch.get_all(engine, container_ref)
    for b in branches:
        if b.name == branch_name:
            return b.reference

    raise Exception('{} was not found in container ref {}.\n'.format(branch_name, container_ref))

def find_bookmark_ref(engine, template_name, container_name, branch_name, bookmark_name):
    bookmarks = bookmark.get_all(engine)
    if container_name and branch_name:
        # container bookmark
        branch_ref = find_branch_ref(engine, template_name, container_name, branch_name)
        for bmark in bookmarks:
            if bmark.name == bookmark_name and bmark.branch == branch_ref and bmark.container_name == container_name and bmark.template_name == template_name:
                return bmark.reference
    else:
        # template bookmark
        for bmark in bookmarks:
            if bmark.name == bookmark_name and bmark.template_name == template_name:
                return bmark.reference

    raise Exception('{} was not found in container {}, branch {}.\n'.format(bookmark_name, container_name, branch_name))


def find_obj_by_name(engine, f_class, obj_name, active_branch=False):
    """
    Function to find objects by name and object class, and return object's 
    reference as a string
    engine: A Delphix engine session object
    f_class: The objects class. I.E. database or timeflow.
    obj_name: The name of the object
    active_branch: Default = False. If true, return list containing
                   the object's reference and active_branch. Otherwise, return 
                   the reference.
    """

    return_list = []

    try:
        all_objs = f_class.get_all(engine)
    except AttributeError as e:
        raise Exception('Could not find reference for object class'
                            '{}.\n'.format(e))
    for obj in all_objs:
        if obj.name == obj_name:
            print("Found object named %s\n." % obj.name)

            if active_branch is False:
                return(obj)

            #This code is for JS objects only.
            elif active_branch is True:
                return_list.append(obj.reference)
                return_list.append(obj.active_branch)
                return(return_list)

            return obj

    #If the object isn't found, raise an exception.
    raise Exception('{} was not found on engine {}.\n'.format(
        obj_name, engine.address))


def get_obj_reference(engine, obj_type, obj_name,
                      container=False):
    """
    Return the reference for the provided object name
    engine: A Delphix engine object.
    results: List containing object name
    search_str (optional): string to search within results list
    container (optional): search for container instead of name
    """

    results = obj_type.get_all(engine)

    for result in results:
        print("Found object named %s with reference %s." % (result.name, result.reference))
        if container is False:
            if result.name == obj_name:
                return result.reference
        else:
            if result.container == obj_name:
                return result.reference

    raise Exception('Reference not found for {}.'.format(obj_name))


def find_obj_name(engine, f_class, obj_reference):
    """
    Return the obj name from obj_reference

    engine: A Delphix engine object.
    f_class: The objects class. I.E. database or timeflow.
    obj_reference: The object reference to retrieve the name
    """
    try:
        obj_name = f_class.get(engine, obj_reference)
        return obj_name.name

    except RequestError as e:
        raise Exception(e)

    except (JobError, HttpError) as e:
        raise Exception(e.message)


def find_dbrepo(engine, install_type, f_environment_ref, f_install_path):
    """
    Function to find database repository objects by environment reference and
    install path, and return the object's reference as a string
    You might use this function to find Oracle and PostGreSQL database repos.
    engine: Virtualization Engine Session object
    install_type: Type of install - Oracle, ASE, SQL
    f_environment_ref: Reference of the environment for the repository
    f_install_path: Path to the installation directory.
    return: delphixpy.web.vo.SourceRepository object
    """

    all_objs = repository.get_all(engine, environment=f_environment_ref)
    for obj in all_objs:
        print(str(obj))
        if 'OracleInstall' == install_type:
            if (obj.type == install_type and
                    obj.installation_home == f_install_path):
                return obj

        elif 'MSSqlInstance' == install_type:
            if (obj.type == install_type and
                    obj.installation_path == f_install_path):
                return obj

        elif 'MySQLInstall' == install_type:
            if (obj.type == install_type and
                    obj.installation_path == f_install_path):
                return obj

        else:
            raise Exception('No Repo match found for type {}.\n'.format(
                install_type))


def find_database_source_from_name(engine, db_name):
    all_sources = source.get_all(engine)
    for src in all_sources:
        if src.name == db_name:
            print("Found source named %s for database %s." % (src.reference, db_name))
            return src
    print("No source found for database named %s." % db_name)