#
# Copyright 2024 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import json
from xlrelease.HttpRequest import HttpRequest
from dynatrace.util import error

class DynatraceServer(object):

    def __init__(self, httpConnection):
        httpConnection['url'] = httpConnection['url'].strip("/")
        self.httpRequest = HttpRequest(httpConnection)
        self.apiKey = httpConnection['apiKey']
        self.headers = self._get_headers()
    
    def _get_headers(self):
        return {"Accept": "application/json", "Content-Type": "application/json", "Authorization": 'Api-Token ' + self.apiKey}
    

    def getHttpMonitorExecution(self, monitorId, monitorResult, locationId):
        getExecutionUrl = 'api/v2/synthetic/execution/%s/%s' % (monitorId, monitorResult)

        if locationId is not None:
            getExecutionUrl += '?locationId=%s' % (locationId) 
        response = self.httpRequest.get(getExecutionUrl, headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while fetching executions", response)
        response_body = json.loads(response.response)
        return response_body
    
    def getMonitors(self, type=None):
        getMonitorsUrl = 'api/v1/synthetic/monitors'
        if type is not None:
            getMonitorsUrl += '?type=%s' % (type)
        response = self.httpRequest.get(getMonitorsUrl, headers=self.headers)
        if response.getStatus() != 200:
            return []
        response_body = json.loads(response.response)
        return response_body['monitors']
    
    def getLocations(self):
        getLocationsUrl = 'api/v2/synthetic/locations'
        response = self.httpRequest.get(getLocationsUrl, headers=self.headers)
        if response.getStatus() != 200:
            return []
        response_body = json.loads(response.response)
        return response_body['locations']
    
    def getMonitorDetails(self, monitorId):
        getMonitorDetailsUrl = 'api/v1/synthetic/monitors/%s' % (monitorId)
        response = self.httpRequest.get(getMonitorDetailsUrl, headers=self.headers)
        if response.getStatus() != 200:
            return []
        response_body = json.loads(response.response)
        return response_body
    
    
    def triggerOnDemandExecutions(self, monitorID, failOnPerformanceThreshold, location, failOnSslWarning, executionCount=1):
        triggerOnDemandUrl= 'api/v2/synthetic/executions/batch'
        data = {
                "failOnPerformanceIssue": failOnPerformanceThreshold,
                "failOnSslWarning": failOnSslWarning,
                "monitors": [
                    {
                        "executionCount":executionCount,
                        "monitorId": monitorID
                    }
                ]
            }
        if location:
            data["monitors"][0]["locations"] = [location] 
        requestBody=json.dumps(data)
        response=self.httpRequest.post(triggerOnDemandUrl, requestBody, headers=self.headers)
        if response.getStatus() != 201:
            error("An error occurred while fetching executions", response)
        responseBody = json.loads(response.response)
        return responseBody
    
    def getBatchStatus(self, batchID):
        batchDetailsUrl='api/v2/synthetic/executions/batch/%s' % (batchID)
        response=self.httpRequest.get(batchDetailsUrl, headers=self.headers)
        if response.getStatus() != 200:
            error("An error occurred while fetching batch status", response)
        response=response.getResponse()
        response=json.loads(response)
        return response
    
    def createSyntheticMonitor(self, requestBody):
        createMonitorUrl='api/v1/synthetic/monitors'
        requestBody=json.dumps(requestBody)
        response=self.httpRequest.post(createMonitorUrl, requestBody, headers=self.headers)
        if response.getStatus() !=200:
            error("An error occured while creating monitor: ", response)
        response=response.getResponse()
        response=json.loads(response)
        return response["entityId"]

    def toggleMonitorAlert(self, monitorId, globalOutage, localOutage, monitorConfig):
        toggleMonitorUrl = 'api/v1/synthetic/monitors/%s' % (monitorId)

        if 'anomalyDetection' in monitorConfig and 'outageHandling' in monitorConfig['anomalyDetection']:
            outageHandling = monitorConfig['anomalyDetection']['outageHandling']
            if 'globalOutage' in outageHandling:
                outageHandling['globalOutage'] = globalOutage
            if 'localOutage' in outageHandling:
                outageHandling['localOutage'] = localOutage
        
        requestBody = json.dumps(monitorConfig)
        response = self.httpRequest.put(toggleMonitorUrl, requestBody, headers=self.headers)
        if response.getStatus() != 204:
            error("Failed to toggle outage state", response)

    def toggleMonitorState(self, monitorId, monitorState, monitorConfig):
        toggleMonitorUrl = 'api/v1/synthetic/monitors/%s' % (monitorId)

        if 'enabled' in monitorConfig:
            monitorConfig['enabled'] = monitorState
        
        requestBody = json.dumps(monitorConfig)
        response = self.httpRequest.put(toggleMonitorUrl, requestBody, headers=self.headers)
        if response.getStatus() != 204:
            error("Failed to toggle monitor state", response)
