#
# Copyright (c) 2019. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries and licensors.
#

import time
from datetime import datetime
from math import ceil

from fortifyOnDemand import FortifyOnDemandService
from fortifyOnDemand.concurrency.FortifyConcurrentExecutorService import FortifyConcurrentExecutorService
from fortifyOnDemand.concurrency.FortifyOnDemandComplianceTasks import GetApplicationReleaseTask, LIMIT, GetReleaseVulnerabilityFiltersTask

global fortifyOnDemandServer, projectName, tile, standards, timeFrame, dateFrom, dateTo


def filter_by_date(from_date, to_date):
    def do_filter(date):
        date = time.mktime(datetime.strptime(date, '%Y-%m-%dT%H:%M:%S.%f').timetuple())
        return from_date <= date <= to_date

    return do_filter


# Validate properties
missedProperties = []
if not fortifyOnDemandServer:
    missedProperties.append('Fortify server')
if not projectName:
    missedProperties.append('Application name')
if not standards:
    missedProperties.append('Standards')
if not timeFrame:
    missedProperties.append('Time frame')
if not tile.hasProperty("availableStandards"):
    missedProperties.append('Available standards')

if missedProperties:
    raise Exception("{} must be provided".format(', '.join(missedProperties)))

fortify_on_demand_standards = tile.getProperty("availableStandards")

# Load application releases =========================================
application_release = FortifyOnDemandService.get_releases_by_application_name(fortifyOnDemandServer, application_name=projectName, limit=LIMIT)
application_releases = application_release['items']
total_pages = pages = int(ceil(application_release['totalCount'] / float(LIMIT)))

executor = FortifyConcurrentExecutorService()
for offset in range(1, total_pages):
    executor.add(GetApplicationReleaseTask(fortifyOnDemandServer, projectName, offset * LIMIT, LIMIT))
futures = executor.execute()

for future in futures:
    result = future.get()
    if result.exception is not None:
        raise result.exception
    application_releases += result.result

# Filter releases by date
filter_by_date_func = filter_by_date(timeFrame.getStartDate(dateFrom) / 1000.0, timeFrame.getEndDate(dateTo) / 1000.0)
application_releases = list(filter(lambda release: filter_by_date_func(release['releaseCreatedDate']), application_releases))

# Add releaseUrl to application_releases
for release in application_releases:
    release['releaseUrl'] = "%s/Releases/%s/Overview" % (fortifyOnDemandServer.get('url').replace('api.', '').strip('/'), release['releaseId'])
#  / Load application releases =======================================

# Load release vulnerabilities ===================================
executor = FortifyConcurrentExecutorService()
for release in application_releases:
    executor.add(GetReleaseVulnerabilityFiltersTask(fortifyOnDemandServer, fortify_on_demand_standards, release))
futures = executor.execute()

data = []
for future in futures:
    result = future.get()
    if result.exception is not None:
        raise result.exception
    data.append(result.result)
# / Load release vulnerabilities =================================
