#
# Copyright 2020 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import sets
import json
from xlrelease.HttpRequest import HttpRequest
from actions import get_proxy

HTTP_SUCCESS = sets.Set([200, 201, 204])


class GithubRestClient:

    def __init__(self, gitServer, username=None, token=None, organization=None):
        self.username = gitServer["loginOrToken"]
        if username:
            self.username = username
        if organization:
            self.username = organization
        self.token = gitServer["password"]
        if token:
            self.token = token
        proxyHost, proxyPort, proxyUsername, proxyPassword = get_proxy()
        server = {'url': gitServer['baseUrl'], 'proxyHost': proxyHost, 'proxyPort': proxyPort,'proxyUsername': proxyUsername, 'proxyPassword': proxyPassword}
        self.httpRequest = HttpRequest(server, verify=False)
        self.headers = {'Authorization': 'token %s' % self.token, 'Accept': 'application/vnd.github+json'}

    def getWorkflowsList(self, repositoryName):
        url = '/repos/' + self.username + '/' + repositoryName + '/actions/workflows'
        response = self.httpRequest.get(url,  headers=self.headers)
        return self.get_results(response)

    def getWorkflowDetails(self, repositoryName, workflowIdFilename):
        url = '/repos/' + self.username + '/' + repositoryName + '/actions/workflows/'+workflowIdFilename
        response = self.httpRequest.get(url,  headers=self.headers)
        return self.get_results(response)

    def enableDisableWorkflow(self, repositoryName, workflowIdFilename, action):
        url = '/repos/' + self.username + '/' + repositoryName + '/actions/workflows/'+workflowIdFilename+"/"+action
        response = self.httpRequest.put(url,  body='', headers=self.headers)
        return self.get_results(response)

    def createDispatchEvent(self, repositoryName, workflowIdFilename, body):
        url = '/repos/' + self.username + '/' + repositoryName + '/actions/workflows/' + workflowIdFilename + '/dispatches'
        response = self.httpRequest.post(url, body=json.dumps(body), headers=self.headers)
        return self.get_results(response)

    def getWorkflowRunsList(self, repositoryName):
        url = '/repos/' + self.username + '/' + repositoryName + '/actions/runs?per_page=1'
        response = self.httpRequest.get(url,  headers=self.headers)
        return self.get_results(response)

    def getWorkflowRunDetails(self, repositoryName, runId):
        url = '/repos/' + self.username + '/' + repositoryName + '/actions/runs/'+ runId
        response = self.httpRequest.get(url,  headers=self.headers)
        return self.get_results(response)

    def get_results(self, response):
        if response.getStatus() in HTTP_SUCCESS:
            if response.getResponse() is not None:
                return json.loads(response.getResponse())
        else:
            raise Exception(
                "Error from server, HTTP Return: %s, content %s" % (response.getStatus(), response.getResponse()))
