# Copyright (c) 2021. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

import os
import json
import yaml
from urllib import quote
from xlrelease.HttpRequest import HttpRequest
from com.jayway.jsonpath import JsonPath

PAGE_SIZE = 100


class Client(object):
    def __init__(self):
        return

    def get_client(self):
        return Client()

    def get_gitlab_server(self, variables):
        gitlab_server = variables["gitlab_server"]
        if gitlab_server is None:
            raise Exception("No GitLab Server provided!")
        return gitlab_server

    def get_gitlab_api_key(self, variables):
        gitlab_server = self.get_gitlab_server(variables)
        if not variables["api_key"]:
            if variables["gitlab_server"]["api_key"]:
                return gitlab_server["api_key"]
            else:
                raise Exception("API Key Not Set!")
        else:
            return variables["api_key"]

    def handle_response(self, response, err_message=None):
        if response.status < 400:
            if response.response:
                return json.loads(response.response)
            else:
                return {'status':'success'}
        else:
            if err_message:
                raise Exception("{}: {}".format(err_message, response.errorDump()))
            else:
                raise Exception("Unexpected Error: {}".format(response.errorDump()))

    def build_projects_endpoint(self, url, variables):
        return "/api/v4/projects{0}&private_token={1}".format(
            url, self.get_gitlab_api_key(variables),
        )

    def build_projects_endpoint_1(self, url, variables):
        return "/api/v4/projects{0}?private_token={1}".format(
            url, self.get_gitlab_api_key(variables),
        )

    def build_projects_pipeline_endpoint(self, project):
        return "/api/v4/projects/{0}/trigger/pipeline".format(project)

    def build_content(self, params):
        content = ""
        for key in params.keys():
            value = params[key]
            if value is not None:
                content = "{0}&{1}={2}".format(content, key, value)
        return content

    def get_request(self, variables):
        gitlab_server = self.get_gitlab_server(variables)
        return HttpRequest(gitlab_server)
    
    def lookup_helper(self, variables, branch_name):
        if not branch_name:
            branch_name = ""
        endpoint = "/api/v4/projects/{0}/repository/branches?private_token={1}&per_page=100&search={2}".format(
            variables["project_id"], self.get_gitlab_api_key(variables), branch_name
        )
        data = self.handle_response(
            self.get_request(variables).get(endpoint, contentType="application/json")
        )
        return data
         
    def gitlab_createmergerequest(self, variables):
        content ={ "source_branch": variables["source_branch"], "target_branch": variables["target_branch"],
            "title": variables["title"], "remove_source_branch": variables["remove_source_branch"], "target_project_id": variables["target_project_id"]}
        content = json.dumps(content)
        data = self.handle_response(
            self.get_request(variables).post(
                self.build_projects_endpoint(
                    "/{}/merge_requests?".format(variables["project_id"]), variables
                ),
                content,
                contentType="application/json",
            )
        )
        return {"merge_id": str(data.get("iid"))}

    def get_merge_request_status(self, variables):
        endpoint = self.build_projects_endpoint(
            "/{0}/merge_requests?iids[]={1}".format(
                variables["project_id"], variables["merge_id"]
            ),
            variables,
        )
        responseDict = self.handle_response(
            self.get_request(variables).get(endpoint,contentType="application/json"))

        mr = responseDict[0]
        response = {
            'state' : mr['state'],
            'detailed_merge_status' : mr['detailed_merge_status']
        }
        return response

    def gitlab_acceptmergerequest(self, variables):
        if variables['waitForApproval']:
            mr_status = self.get_merge_request_status(variables)
            if mr_status['detailed_merge_status'] == 'not_approved':
                if variables['currentRetryCount'] >=  variables['pollingRetryCount']:
                    raise Exception("Polling retries exhausted waiting for Merge request approval.")
                else:
                    return {"result": "retry",
                            "status": mr_status['detailed_merge_status']}

        responseDict = self.handle_response(
            self.get_request(variables).put(
                self.build_projects_endpoint(
                    "/{0}/merge_requests/{1}/merge?".format(
                        variables["project_id"], variables["merge_id"]
                    ),
                    variables,
                ),
                "",
                contentType="application/json",
            )
        )
        if responseDict['state'] != "merged":
            if responseDict["merge_error"]:
                raise Exception("Error Merging the request : ",responseDict["merge_error"])
            else:
                raise Exception("Unable to Merge the request")

        print("Merged Merge Request number: {0}".format(str(variables["merge_id"])))
        return {"result": "merged"}

    def gitlab_approvemergerequest(self, variables):
        data = self.handle_response(
            self.get_request(variables).post(
                self.build_projects_endpoint(
                    "/{0}/merge_requests/{1}/approve?".format(variables["project_id"],
                                                          variables["merge_id"]),
                                                                variables),
                "",
                contentType="application/json",
            )
        )
        return {"status": "Approved"}

    def gitlab_commentmergerequest(self, variables):
        data = self.handle_response(
            self.get_request(variables).post(
                self.build_projects_endpoint(
                    "/{0}/merge_requests/{1}/notes?body={2}".format(variables["project_id"],
                                                              variables["merge_id"], variables["comment"]),
                    variables),
                "",
                contentType="application/json",
            )
        )
        print("Added comment to MergeRequest %s" % (variables["merge_id"]))
        return {"status": "success"}

    def gitlab_closemergerequest(self, variables):
        if variables['comment']:
            self.gitlab_commentmergerequest(variables)

        data = self.handle_response(
            self.get_request(variables).put(
                self.build_projects_endpoint(
                    "/{0}/merge_requests/{1}?state_event=close".format(variables["project_id"],
                                                              variables["merge_id"]),
                    variables),
                "",
                contentType="application/json",
            )
        )
        print("Closed MergeRequest %s" % (variables["merge_id"]))
        return {"status": "Approved"}

    def gitlab_deletetag(self, variables):
        data = self.handle_response(
            self.get_request(variables).delete(
                self.build_projects_endpoint_1(
                    "/{0}/repository/tags/{1}".format(variables["project_id"],variables["tag_name"]),
                    variables),
                contentType="application/json",
            )
        )
        print("Project: %s - Tag: %s deleted" % (variables["project_id"], variables["tag_name"]))
        return {"status": "deleted"}

    def filter_project_on_namespace(self, data, namespace):
        if namespace is None:
            return {"project_id": ""}
        for project in data:
            if namespace in project["name_with_namespace"]:
                return {"project_id": str(project["id"])}
        return {"project_id": ""}

    def gitlab_querydata(self, variables):
        url= variables["endpoint"]
        queryParameters= variables.get("queryParams", None)
        context = url+"?private_token={0}".format(
            self.get_gitlab_api_key(variables))
        if queryParameters:
            context +=  self.build_content(queryParameters)
        data = self.handle_response(
            self.get_request(variables).get(
                context, contentType="application/json"
            )
        )
        jsoncontext = JsonPath.parse(data)
        result = {}
        result["value"] = str(jsoncontext.read(variables["path_spec"]))
        return result

    def gitlab_querysecuredata(self, variables):
        # The returned value is handled as a password in the XLR user interface, but the API call is the same as gitlab_querydata
        return self.gitlab_querydata(variables)

    def gitlab_queryproject(self, variables):
        project_name = variables["project_name"]
        if not project_name:
            raise ValueError("Project name must be provided")
        target_namespace = variables.get("namespace")
        search_url = "?search={}".format(project_name)
        endpoint = self.build_projects_endpoint(search_url, variables)
        response = self.get_request(variables).get(
            endpoint,
            contentType="application/json"
        )
        data = self.handle_response(response)

        exact_matches = []

        for project in data:
            # --- FILTER 1: Exact Name Match ---
            if project.get("name") != project_name:
                continue

            # --- FILTER 2: Namespace Match (Ignoring Spaces) ---
            if target_namespace:
                full_path = project.get("name_with_namespace", "")
                if "/" in full_path:
                    actual_ns_part, _ = full_path.rsplit('/', 1)
                else:
                    actual_ns_part = ""

                clean_actual = actual_ns_part.replace(" ", "")
                clean_target = target_namespace.replace(" ", "").rstrip("/")

                if clean_actual == clean_target:
                    exact_matches.append(project)
            else:
                exact_matches.append(project)

        # --- RETURN OR RAISE ---
        if len(exact_matches) == 1:
            return {"project_id": str(exact_matches[0]["id"])}
        elif len(exact_matches) > 1:
            raise ValueError(
                "Ambiguous: Found {0} projects named '{1}'. "
                "Please check your namespace.".format(len(exact_matches), project_name))
        else:
            msg = "Project '{0}' not found".format(project_name)
            if target_namespace:
                msg += " in namespace '{0}'".format(target_namespace)
            raise ValueError(msg)

    def gitlab_querymergerequests(self, variables):
        endpoint = self.build_projects_endpoint(
            "/{0}/merge_requests?state={1}".format(
                variables["project_id"], variables["state"]
            ),
            variables,
        )
        # Sorting and filtering merge request results
        if variables["sorting"] == "Creation Datetime Descending":
            endpoint += "&order_by=created_at&sort=desc"
        if variables["sorting"] == "Creation Datetime Ascending":
            endpoint += "&order_by=created_at&sort=asc"
        if variables["sorting"] == "Last Update Datetime Descending":
            endpoint += "&order_by=updated_at&sort=desc"
        if variables["sorting"] == "Last Update Datetime Ascending":
            endpoint += "&order_by=updated_at&sort=asc"
        if variables["simple_view"]:
            endpoint += "&view=simple"
        if variables["source_branch"] is not None:
            endpoint += "&source_branch={}".format(variables["source_branch"])
        if variables["target_branch"] is not None:
            endpoint += "&target_branch={}".format(variables["target_branch"])
        if variables["milestone"] is not None:
            endpoint += "&milestone={}".format(variables["milestone"])
        # Pagination
        merge_requests = []
        # Calculate page sizes using max PAGE_SIZE results per page (GitLab limit) and the user-specified results_limit
        result_set_sizes = [
            min(variables["results_limit"] - i, PAGE_SIZE)
            for i in range(0, variables["results_limit"], PAGE_SIZE)
        ]
        for page_num, result_set_size in enumerate(result_set_sizes, 1):
            endpoint_page = endpoint + "&per_page={0}&page={1}".format(
                PAGE_SIZE, page_num
            )
            response = self.get_request(variables).get(endpoint_page, contentType="application/json")
            merge_requests_set = self.handle_response(response)
            if merge_requests_set == []:  # no more results to pull
                break
            else:  # pull results based on expected results_limit count for that page
                merge_requests += merge_requests_set[0:result_set_size]
        return {"merge_requests": str(json.dumps(merge_requests))}

    def gitlab_createtag(self, variables):
        content = {
            "tag_name": variables["tag_name"],
            "ref": variables["ref"],
            "message": variables["message"]
        }
        content = json.dumps(content)
        data = self.handle_response(
            self.get_request(variables).post(
                self.build_projects_endpoint(
                    "/{}/repository/tags?".format(variables["project_id"]), variables
                ),
                content,
                contentType="application/json",
            )
        )
        return {"commit_id": str(data["commit"]["id"])}

    def gitlab_createbranch(self, variables):
        content = {"branch": variables["branch"], "ref": variables["ref"]}
        content = json.dumps(content)
        data = self.handle_response(
            self.get_request(variables).post(
                self.build_projects_endpoint(
                    "/{}/repository/branches?".format(variables["project_id"]),
                    variables,
                ),
                content,
                contentType="application/json",
            )
        )
        return {"commit_id": str(data["commit"]["id"])}

    def gitlab_triggerpipeline(self, variables):
        endpoint = "/api/v4/projects/{0}/ref/{1}/trigger/pipeline?token={2}".format(
            variables["project_id"], variables["ref"], variables["token"]
        )
        for key, value in variables["variables"].iteritems():
            endpoint += "&variables[{0}]={1}".format(key, value)

        data = self.handle_response(
            self.get_request(variables).post(endpoint, "", contentType="application/json")
        )
        print("[Pipeline #{0}]({1})".format(data["id"], data["web_url"]))
        status = {"pipeline_id": str(data["id"]), "status": str(data["status"])}
        return status

    def gitlab_createpipeline(self, variables, gitlab_server):
        endpoint = "/api/v4/projects/{0}/pipeline?ref={1}".format(
            variables["project_id"], variables["ref"]
        )
        headers = {
            'Content-Type': "application/json",
            'PRIVATE-TOKEN': gitlab_server["api_key"]
        }
        content = {
            "ref": variables["ref"],
            "variables": []
        }
        for key, value in variables["variables"].items():
            content["variables"].append({"key": key, "value": value})
        content = json.dumps(content)
        data = self.handle_response(
            self.get_request(variables).post(endpoint, content, contentType="application/json", headers=headers),
            "Create Pipeline failed"
        )
        print("[Pipeline Created #{0}]({1})".format(data["id"], data["web_url"]))
        status = {"pipeline_id": str(data["id"]), "pipeline_web_url": str(data["web_url"])}
        return status

    def gitlab_pipeline_status(self, variables):
        pipeline_id = variables.get("pipeline_id")

        if pipeline_id is None:
            endpoint = "/api/v4/projects/{0}/pipelines/?private_token={1}&sha={2}&ref={3}".format(
                variables["project_id"],self.get_gitlab_api_key(variables), variables["commit_id"], variables["ref"]
            )
        elif pipeline_id is not None:
            pipeline_id = variables["pipeline_id"]
            endpoint = "/api/v4/projects/{0}/pipelines/{1}?private_token={2}".format(
                variables["project_id"], pipeline_id, self.get_gitlab_api_key(variables)
            )
        else:
            print("check input properties/variables for pipeline status task")
        data = self.handle_response(self.get_request(variables).get(endpoint, contentType="application/json"
        ))
        if type(data) is dict:
            status = {
                "pipeline_id": "{0}".format(data.get("id")),
                "status": data.get("status"),
                "web_url": data.get("web_url"),
            }
        else:
            status = {
                "pipeline_id": "{0}".format(data[0].get("id")),
                "status": data[0].get("status"),
                "web_url": data[0].get("web_url"),
            }
        return status

    def gitlab_branch_statuses(self, variables):
        endpoint = "/api/v4/projects/{0}/repository/branches?private_token={1}".format(
            variables["project_id"], self.get_gitlab_api_key(variables)
        )
        branches = self.handle_response(self.get_request(variables).get(endpoint, contentType="application/json"))
        # build a map of the commit ids for each branch
        latest_commits = {}
        for branch in branches:
            if not variables["branchName"] or branch["name"] == variables["branchName"]:
                branch_id = branch["name"]
                last_commit = branch["commit"]["id"]
                latest_commits[branch_id] = last_commit
        return latest_commits

    def gitlab_revertCommit(self, variables):
        branch_name = variables["branchName"]
        commit_id = variables["commitId"]
        if not commit_id:
            commit_id = self.gitlab_branch_statuses(variables)[branch_name]
        endpoint = "/api/v4/projects/{0}/repository/commits/{1}/revert?private_token={2}".format(
            variables["project_id"], str(commit_id), self.get_gitlab_api_key(variables))
        data = {"branch": branch_name}
        content = json.dumps(data)
        revert_commit = self.handle_response(self.get_request(variables).post(endpoint, content, contentType="application/json"), "Revert Commit Failed.")
        return revert_commit

    def gitlab_tag_statuses(self, variables):
        endpoint = "/api/v4/projects/{0}/repository/tags?private_token={1}&order_by={2}&sort={3}".format(
            variables["project_id"],self.get_gitlab_api_key(variables),variables["order_by"],variables["sort"]
        )
        if variables["search"] not in [None, ""]:
            endpoint += "&search={0}".format(variables["search"])
        return self.handle_response(self.get_request(variables).get(endpoint, contentType="application/json"))

    def gitlab_createproject(self, variables):
        proj_spec = {
            "name": variables["project_name"],
            "path": variables["path"],
            "visibility": variables["visibility"]
        }
        for optional_spec in ["namespace", "description", "import_url"]:
            if optional_spec in variables.keys():
                api_attribute = (
                    "namespace_id" if optional_spec == "namespace" else optional_spec
                )
                proj_spec[api_attribute] = variables[optional_spec]
        endpoint = "/api/v4/projects?private_token={0}".format(
            self.get_gitlab_api_key(variables)
        )
        content = json.dumps(proj_spec)
        data = self.handle_response(
            self.get_request(variables).post(endpoint, content, contentType="application/json")
        )
        return {"project_id": str(data["id"])}

    def gitlab_creategroup(self, variables):
        group_spec = {
            "name": variables["group_name"],
            "path": variables["path"],
            "visibility": variables["visibility"],
        }
        for optional_spec in ["description", "parent_id"]:
            if optional_spec in variables.keys():
                group_spec[optional_spec] = variables[optional_spec]
        # content = Client.build_content(group_spec)
        content = json.dumps(group_spec)
        endpoint = "/api/v4/groups?private_token={0}".format(
            self.get_gitlab_api_key(variables)
        )
        data = self.handle_response(
            self.get_request(variables).post(endpoint, content, contentType="application/json")
        )
        return {"group_id": str(data["id"])}

    def gitlab_querycommits(self, variables):
        endpoint = "/api/v4/projects/{0}/repository/commits?private_token={1}".format(
            variables["project_id"], self.get_gitlab_api_key(variables)
        )
        if variables["branch"] is not None:
            endpoint += "&" + "ref_name=" + variables["branch"]
        # Pagination
        commits = []
        # Calculate page sizes using max PAGE_SIZE results per page (GitLab limit) and the user-specified results_limit
        result_set_sizes = [
            min(variables["results_limit"] - i, PAGE_SIZE)
            for i in range(0, variables["results_limit"], PAGE_SIZE)
        ]
        for page_num, result_set_size in enumerate(result_set_sizes, 1):
            endpoint_page = endpoint + "&per_page={0}&page={1}".format(
                PAGE_SIZE, page_num
            )
            response = self.get_request(variables).get(endpoint_page, contentType="application/json")
            commits_set = self.handle_response(response)
            if commits_set == []:  # no more results to pull
                break
            else:  # pull results based on expected results_limit count for that page
                commits += commits_set[0:result_set_size]
        return {"commits": str(json.dumps(commits))}

    def gitlab_querytags(self, variables):
        endpoint = "/api/v4/projects/{0}/repository/tags?private_token={1}&order_by={2}&sort={3}".format(
            variables["project_id"], self.get_gitlab_api_key(variables), variables["order_by"], variables["sort_order"]
        )
        if variables["search"] not in [None, ""]:
            endpoint += "&search={0}".format(variables["search"])
        # Pagination
        tags = []
        # Calculate page sizes using max PAGE_SIZE results per page (GitLab limit) and the user-specified results_limit
        result_set_sizes = [
            min(variables["results_limit"] - i, PAGE_SIZE)
            for i in range(0, variables["results_limit"], PAGE_SIZE)
        ]
        for page_num, result_set_size in enumerate(result_set_sizes, 1):
            endpoint_page = endpoint + "&per_page={0}&page={1}".format(
                PAGE_SIZE, page_num
            )
            response = self.get_request(variables).get(endpoint_page, contentType="application/json")
            tags_set = self.handle_response(response)
            if tags_set == []:  # no more results to pull
                break
            else:  # pull results based on expected results_limit count for that page
                tags += tags_set[0:result_set_size]
        return {"tags": str(json.dumps(tags))}

    def gitlab_querypipelines(self, variables):
        endpoint = "/api/v4/projects/{0}/pipelines?private_token={1}".format(
            variables["project_id"], self.get_gitlab_api_key(variables)
        )
        data = self.handle_response(self.get_request(variables).get(endpoint, contentType="application/json"))
        return {"pipelines": str(json.dumps(data))}

    def gitlab_querypipeline(self, variables, id):
        endpoint = "/api/v4/projects/{0}/pipelines/{1}?private_token={2}".format(
            variables["project_id"], id, self.get_gitlab_api_key(variables)
        )
        data = self.handle_response(self.get_request(variables).get(endpoint, contentType="application/json"))
        return {"pipeline": str(json.dumps(data))}

    def gitlab_createprojectwebhook(self, variables):
        content_params = [
            "url",
            "push_events",
            "issues_events",
            "confidential_issues_events",
            "merge_requests_events",
            "tag_push_events",
            "note_events",
            "job_events",
            "pipeline_events",
            "wiki_page_events",
            "enable_ssl_verification",
            "token",
        ]
        webhook = {}
        for var_key in variables.keys():
            if var_key in content_params:
                webhook[var_key] = variables[var_key]
        content = json.dumps(webhook)
        data = self.handle_response(
            self.get_request(variables).post(
                self.build_projects_endpoint(
                    "/{}/hooks?".format(variables["project_id"]), variables
                ),
                content,
                contentType="application/json",
            )
        )
        return {"hook_id": str(data["id"])}

    def gitlab_runjob(self, variables):
        endpoint = "/api/v4/projects/{}/jobs/{}/play?private_token={}".format(
        variables["projectId"], variables["jobId"], self.get_gitlab_api_key(variables)
        )

        content = ""
        if variables.get("variables"):
            body = {
                "job_variables_attributes": [
                    {"key": k, "value": v} for k, v in variables["variables"].items()
                ]
            }
            content = json.dumps(body)

        response = self.get_request(variables).post(endpoint, content, contentType="application/json")
        return self.handle_response(response)
       
          

    def gitlab_runjob_status(self, variables):  
            endpoint = "/api/v4/projects/{0}/jobs/{1}?private_token={2}".format(
                variables["projectId"], variables["jobId"], self.get_gitlab_api_key(variables)
            )
            data = self.handle_response(
                self.get_request(variables).get(endpoint, contentType="application/json")
         )
            return data
    
    def gitlab_deletebranch(self, variables):
            endpoint = "/api/v4/projects/{0}/repository/branches/{1}?private_token={2}".format(
                variables["project_id"], variables["branch_name"], self.get_gitlab_api_key(variables)
            )
            data = self.handle_response(
                self.get_request(variables).delete(endpoint, contentType="application/json")
            )

    def gitlab_queryfiledata(self, variables):
        file_path = variables.get("file_path")
        if not file_path:
            raise ValueError("File path is not configured")
        file_path = quote(file_path.replace("\\", "/").strip("/"), safe='')

        file_extension = os.path.splitext(file_path)[1].lstrip(".").lower()
        if file_extension not in ["json", "yml", "yaml"]:
            raise ValueError("Unsupported file format: %s" % file_extension)
        json_format = file_extension == "json"

        project_id = variables.get("project_id")
        if not project_id:
            raise ValueError("Project ID is not configured")

        endpoint = self.build_projects_endpoint_1("/{0}/repository/files/{1}/raw".format(project_id, file_path), variables)
        revision = variables.get("revision")
        if revision:
            endpoint += "&ref={0}".format(revision)
        response = self.get_request(variables).get(endpoint, contentType="application/json")
        if response.status != 200:
            raise Exception('%s %s:%s' % ("Failed to retrieve file data from GitLab", response.getStatus(), response.getResponse()))

        data = json.loads(response.response) if json_format else yaml.safe_load(response.response)
        format_value = lambda d: json.dumps(d, indent=4) if json_format else yaml.safe_dump(d, indent=4)
        path_spec = variables.get("path_spec")
        if path_spec:
            value = JsonPath.parse(data).read(path_spec)
            result_value = format_value(value) if isinstance(value, dict) else str(value)
        else:
            result_value = format_value(data)
        return {"value": result_value}