#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from openshift.api_client import APIClient
import time


class ImageTagHelper(object):
    def __init__(self, python_script):
        self.__python_script = python_script
        self.__client = APIClient(python_script.getProperty("client"))

    def __get_full_image_name(self, project_name, image_name):
        return "{0}{1}{2}".format(project_name if bool(project_name) and bool(project_name.strip()) else '',
                                  '/' if bool(project_name) and bool(project_name.strip()) else '', image_name)

    def __get_source_image_name_with_tag(self):
        return self.__get_full_image_name(self.__python_script.getProperty('sourceProjectName'),
                                          self.__python_script.getProperty('sourceImageNameAndTag'))

    def __get_target_image_name_with_tag(self):
        return self.__get_full_image_name(self.__python_script.getProperty('targetProjectName'),
                                          self.__python_script.getProperty('targetImageNameAndTag'))

    def __get_target_image_name(self):
        return self.__python_script.getProperty('targetImageNameAndTag').split(':')[0]

    def __get_target_image_tag(self):
        return self.__python_script.getProperty('targetImageNameAndTag').split(':')[1]

    def __get_target_namespace(self):
        return self.__python_script.getProperty('targetProjectName') if bool(
            self.__python_script.getProperty('targetProjectName')) else (
            self.__python_script.getProperty('sourceProjectName') if bool(
                self.__python_script.getProperty('sourceProjectName')) else '')

    def tag_image(self):
        self.__client.execute_openshift_command(
            command="{0} {1} {2}".format(self.__python_script.getProperty('command'),
                                         self.__get_source_image_name_with_tag(),
                                         self.__get_target_image_name_with_tag()),
            cmd_params=self.__python_script.getProperty('cmdParams'))

    @staticmethod
    def print_log(msg):
        print "```"
        print msg
        print "```"

    def wait_for_tagged_image_to_be_available(self):
        namespace_filter = "--namespace={0}".format(self.__get_target_namespace()) if bool(
            self.__get_target_namespace()) else ""
        counter = 0
        while True:
            response = self.__client.execute_openshift_command_only(
                command="describe is {0} {1}".format(self.__get_target_image_name(), namespace_filter))
            if self.__get_target_image_tag() in str(response.stdout):
                break
            else:
                ImageTagHelper.print_log("Waiting for tagged image {0} to be available.".format(self.__get_target_image_name_with_tag()))
                time.sleep(2)
                if counter > self.__python_script.getProperty('retryCount'):
                    raise RuntimeError("Image tagging failed.")
                else:
                    counter += 1
        ImageTagHelper.print_log("Image is tagged successfully.")

    def logout(self):
        self.__client.logout()
