#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#
from xlropenshift.client.openshift_client import OpenShiftClient
import openshift.client
import kubernetes.client


class ImageTagHelper(object):
    def __init__(self, python_script, server):
        self.__python_script = python_script
        self.__client = OpenShiftClient(server)
        self.__oapi_client = self.__client.oapi_client()

    def tag_image(self):
        source_image_tag = self.get_source_image_tag_details()
        if self.get_target_image_tag_details():
            self.__oapi_client.patch_namespaced_image_stream_tag(name=self.__get_target_image_name_and_tag(),
                                                                 namespace=self.__get_target_project_name(),
                                                                 body=self.__create_image_stream_tag_body(
                                                                     source_image_tag))
        else:
            self.__oapi_client.create_namespaced_image_stream_tag(
                namespace=self.__get_target_project_name(),
                body=self.__create_image_stream_tag_body(source_image_tag))
        print "Image tag created from {}/{} to {}/{}".format(self.__get_source_project_name(),
                                                             self.__get_source_image_name_and_tag(),
                                                             self.__get_target_project_name(),
                                                             self.__get_target_image_name_and_tag())

    def __create_image_stream_tag_body(self, source_image_tag):
        tag_from = kubernetes.client.V1ObjectReference(kind='ImageStreamImage',
                                                       namespace=self.__get_source_project_name(),
                                                       name='{}@{}'.format(self.__get_source_image_name(),
                                                                           source_image_tag.image.metadata.name))
        return openshift.client.V1ImageStreamTag(image=source_image_tag.image,
                                                 metadata=kubernetes.client.V1ObjectMeta(
                                                     name=self.__get_target_image_name_and_tag()),
                                                 tag=openshift.client.V1TagReference(name=self.__get_target_image_tag(),
                                                                                     _from=tag_from),
                                                 generation=source_image_tag.generation,
                                                 lookup_policy=source_image_tag.lookup_policy)

    def get_source_image_tag_details(self):
        # RestApi Call /oapi/v1/namespaces/nginx/imagestreamtags
        # If tag object is null it will throw 'Exception during execution: ValueError: Invalid value for tag, must not be None'
        image_tag_response = self.__oapi_client.list_namespaced_image_stream_tag(
            namespace=self.__get_source_project_name())
        for item in image_tag_response.items:
            if item.metadata.name == self.__get_source_image_name_and_tag():
                return item
        raise RuntimeError(
            "Source image tag not found for : {} on project {} ".format(self.__get_source_image_name_and_tag(),
                                                                        self.__get_source_project_name()))

    def get_target_image_tag_details(self):
        image_tag_response = self.__oapi_client.list_namespaced_image_stream_tag(
            namespace=self.__python_script.getProperty('targetProjectName'))
        for item in image_tag_response.items:
            if item.metadata.name == self.__get_target_image_name_and_tag():
                return item

    def __get_source_project_name(self):
        return self.__python_script.getProperty('sourceProjectName')

    def __get_target_project_name(self):
        return self.__python_script.getProperty('targetProjectName') if self.__python_script.getProperty(
            'targetProjectName') else self.__get_source_project_name()

    def __get_source_image_name(self):
        return self.__python_script.getProperty('sourceImageNameAndTag').split(':')[0]

    def __get_source_image_tag(self):
        splits = self.__python_script.getProperty('sourceImageNameAndTag').split(':')
        return splits[1] if (len(splits) > 1) else 'latest'

    def __get_source_image_name_and_tag(self):
        return '{}:{}'.format(self.__get_source_image_name(), self.__get_source_image_tag())

    def __get_target_image_name(self):
        return self.__python_script.getProperty('targetImageNameAndTag').split(':')[0]

    def __get_target_image_tag(self):
        splits = self.__python_script.getProperty('targetImageNameAndTag').split(':')
        return splits[1] if (len(splits) > 1) else 'latest'

    def __get_target_image_name_and_tag(self):
        return '{}:{}'.format(self.__get_target_image_name(), self.__get_target_image_tag())

    @staticmethod
    def print_log(msg):
        print "```"
        print msg
        print "```"

    def logout(self):
        self.__client.logout()
