'use strict';

(function () {

    var XlrTimelineDirectiveControllerInject = ['ColorService'];
    var XlrTimelineDirectiveController = function XlrTimelineDirectiveController(ColorService) {
        var data = {};
        var currentRelease;

        var outDeps = [];
        var inDeps = [];

        var vm = this;
        vm.elapsedWidth = 0;
        vm.plannedWidth = 0;
        vm.estimatedWidth = 0;
        vm.delayWidth = 0;
        vm.timeline = {};
        vm.phases = [];
        vm.timelineHeight = 30;
        vm.phasesMargin = 85;
        vm.outgoingDependencies = [];
        vm.incomingDependencies = [];
        vm.showTimeIndicator = false;

        vm.redrawTimeline = redrawTimeline;
        vm.init = init;

        vm.buildReleaseLink = function (releaseId) {
            return '#/releases/' + releaseId + '/summary';
        };

        var CHARACTER_WIDTH = 8;

        var TIMELINE = {
            QUARTER_OF_HOUR: 15,
            HOURS_IN_A_DAY: 24,
            DAYS_IN_A_WEEK: 7,
            DAYS_IN_MONTH: 30
        };

        var METRICS = {
            MINUTES: 'minutes',
            HOURS: 'hours',
            DAYS: 'days',
            WEEKS: 'weeks',
            MONTHS: 'months'
        };

        var TIMELINE_FORMATS = {
            HOURS: 'HH:00',
            DAYS: 'D/MM/YY',
            MONTHS: 'MMMM YYYY'
        };

        var COMPARE_FORMAT = {
            HOURS: 'YYYY-MM-DD HH',
            DAYS: 'YYYY-MM-DD'
        };

        var STEP_INDENT = 5;
        var day = 24;
        var week = 168;
        var month = 672;

        function redrawTimeline(width, height) {
            if (data && width) {
                vm.timelineWidth = Math.ceil(width);
                vm.timelineHeight = height;
                vm.elapsedWidth = 0;
                vm.plannedWidth = 0;
                vm.delayWidth = 0;
                vm.timeline = {};
                vm.showTimeIndicator = vm.timelineData.timeline.release.status == 'IN_PROGRESS' || vm.timelineData.timeline.release.status == 'PLANNED';
                vm.phases = [];
                drawTimeline(data);
            }
        }

        function init(width, height) {
            data = vm.timelineData.timeline;
            inDeps = vm.timelineData.incomingDependencies;
            outDeps = vm.timelineData.outgoingDependencies;
            redrawTimeline(width, height);
        }

        function buildPhase(phases, release) {
            var currentPhaseX = 0;
            return _.map(phases, function (phase) {
                phase.color = ColorService.computePhaseColor(release, phase);
                var calculatedPhaseX = getPhaseX(phase, vm.timeline);
                if (calculatedPhaseX < currentPhaseX) {
                    phase.x = currentPhaseX;
                    var widthDiff = currentPhaseX - calculatedPhaseX;
                    phase.width = getPhaseWidth(phase, vm.timeline) - widthDiff;
                } else {
                    phase.x = calculatedPhaseX;
                    phase.width = getPhaseWidth(phase, vm.timeline);
                }
                phase.shortTitle = calculateShortTitle(phase.title, phase.width);
                currentPhaseX = phase.x + phase.width;
                return phase;
            });
        }

        function buildTimelinePeriods(startDay, format, timelineScale, containerSize) {
            var timeline = [];
            for (var i = 0; i < containerSize; i++) {
                timeline.push({
                    title: startDay.format(format)
                });
                startDay = startDay.add(1, timelineScale);
            }
            return timeline;
        }

        function buildContainer(startDay, endDay, timelineScale) {
            var container = {};
            var timelineStart = moment(startDay.format(COMPARE_FORMAT.DAYS));
            var timelineEnd = moment(endDay.format(COMPARE_FORMAT.DAYS));
            container.stepUnit = 1;
            container[timelineScale] = [];
            container.size = Math.ceil(timelineEnd.diff(timelineStart, timelineScale, true) + 1);

            switch (timelineScale) {
                case METRICS.HOURS:
                    container.segments = TIMELINE.QUARTER_OF_HOUR;
                    container.minimalMetric = METRICS.MINUTES;
                    container.stepUnit = 4;
                    timelineStart = moment(startDay.format(COMPARE_FORMAT.HOURS));
                    timelineEnd = moment(endDay.format(COMPARE_FORMAT.HOURS));
                    container.size = Math.ceil(timelineEnd.diff(timelineStart, timelineScale, true) + 1);
                    container[timelineScale] = buildTimelinePeriods(startDay, TIMELINE_FORMATS.HOURS, timelineScale, container.size);
                    break;
                case METRICS.DAYS:
                    container.segments = TIMELINE.HOURS_IN_A_DAY;
                    container.minimalMetric = METRICS.HOURS;
                    container[timelineScale] = buildTimelinePeriods(startDay, TIMELINE_FORMATS.DAYS, timelineScale, container.size);
                    break;
                case METRICS.WEEKS:
                    container.segments = TIMELINE.DAYS_IN_A_WEEK;
                    container.minimalMetric = METRICS.DAYS;
                    container[timelineScale] = buildTimelinePeriods(startDay, TIMELINE_FORMATS.DAYS, timelineScale, container.size);
                    break;
                case METRICS.MONTHS:
                    timelineStart = moment().year(startDay.year()).month(startDay.months()).date(startDay.date());
                    timelineEnd = moment(endDay.endOf('month'));
                    container.size = Math.ceil(timelineEnd.diff(timelineStart, timelineScale, true) + 1);
                    container.segments = TIMELINE.DAYS_IN_MONTH;
                    container.minimalMetric = METRICS.DAYS;
                    container[timelineScale] = buildTimelinePeriods(startDay, TIMELINE_FORMATS.MONTHS, timelineScale, container.size);
                    break;
            }

            container.timelineStart = timelineStart;
            container.timelineEnd = timelineEnd;
            container.metric = timelineScale;
            return container;
        }

        function drawTimeline(data) {
            if (!data || data.startDate === null || data.expectedEndDate === null) return;
            var startDay = data.startDate.clone();
            var endDay = data.expectedEndDate.clone();
            var detectMetric = calculateMetrics(startDay, endDay);
            vm.timeline = createTimeline(detectMetric);
            vm.phases = buildPhase(data.phases, data.release);
            vm.elapsedWidth = getElapsedSectionWidth(vm.timeline);
            vm.delayX = getPlannedSectionWidth(data.endDate, vm.timeline);
            vm.estimatedWidth = getPlannedSectionWidth(data.expectedEndDate, vm.timeline) - vm.phases[0].x;
            var delay = getDelaySectionWidth(data.endDate, data.expectedEndDate, vm.timeline) - vm.delayX;
            vm.delayWidth = delay > 0 ? delay : 0;
            vm.incomingDependencies = buildReleasesViews(inDeps);
            vm.outgoingDependencies = buildReleasesViews(outDeps);
            if (vm.isDetailsView) {
                vm.phasesMargin = calculatePhaseMargin();
                vm.timelineHeight = calculateHeight();
            }
        }

        function calculateShortTitle(title, width) {
            var actual = title;
            var maxLengthOfString = width / CHARACTER_WIDTH;

            if (actual.length > maxLengthOfString) {
                actual = actual.substring(0, maxLengthOfString);
                actual = actual + '...';
            }

            return actual;
        }

        function calculateHeight() {
            var initialHeight = 250;
            var inLength = vm.incomingDependencies.length;
            if (inLength > 2) {
                initialHeight += inLength * 30;
            }
            var outLength = vm.outgoingDependencies.length;
            if (outLength > 2) {
                var delta = outLength - 2;
                initialHeight += delta * 30;
            }
            return initialHeight;
        }

        function calculatePhaseMargin() {
            var initialMargin = 120;
            var inLength = vm.incomingDependencies.length;
            if (inLength > 2) {
                var delta = inLength - 2;
                return initialMargin + delta * 30;
            }
            return initialMargin;
        }

        function calculateMetrics(startDay, endDay) {
            var container = {};

            var diff = endDay.diff(startDay, 'hours');

            if (diff < day * 1.5) {
                return buildContainer(startDay, endDay, 'hours');
            }
            if (diff >= day * 1.5 && diff < week * 3) {
                return buildContainer(startDay, endDay, 'days');
            }
            if (diff >= week * 3 && diff < month * 3) {
                return buildContainer(startDay, endDay, 'weeks');
            }
            if (diff >= month * 3) {
                return buildContainer(startDay, endDay, 'months');
            }
            return container;
        }

        function buildReleasesViews(releases) {
            return _.map(releases, function (release) {
                var releaseCoordinateX = getReleaseX(release.startDate, release.estimatedStartDate);
                var releaseWidth = getReleaseWidth(release.startDate, release.estimatedStartDate, release.endDate, release.estimatedEndDate);
                var actualWidth = releaseCoordinateX + releaseWidth > vm.timelineWidth ? vm.timelineWidth - releaseCoordinateX : releaseWidth;
                return {
                    id: release.id,
                    xCord: releaseCoordinateX,
                    width: actualWidth,
                    title: release.title,
                    shortTitle: calculateShortTitle(release.title, actualWidth - 30)
                };
            });
        }

        function getReleaseX(startDate, estimatedStartDate) {
            var start = startDate;
            if (!startDate.isValid()) {
                start = estimatedStartDate;
            }
            if (angular.isDefined(vm.timeline.start) && angular.isDefined(start) && start.isAfter(vm.timeline.start)) {
                var time = start.diff(vm.timeline.start, vm.timeline.minimalMetric, true);
                return time * vm.timeline.step / vm.timeline.unit;
            }
            return 0;
        }

        function getReleaseWidth(startDate, estimatedStartDate, endDate, estimatedDueDate) {
            var start = startDate;
            var end = endDate;
            if (startDate && !startDate.isValid()) {
                start = estimatedStartDate;
            }
            if (endDate && !endDate.isValid()) {
                end = estimatedDueDate;
            }
            if (angular.isDefined(vm.timeline.start) && angular.isDefined(start) && angular.isDefined(end)) {
                var time = end.diff(start, vm.timeline.minimalMetric, true);
                return time * vm.timeline.step / vm.timeline.unit;
            }
            return 0;
        }

        function getPhaseWidth(phase, timeline) {
            if (angular.isDefined(timeline.start) && angular.isDefined(phase.startDate)) {
                var time = phase.endDate.diff(phase.startDate, timeline.minimalMetric, true);
                return time * timeline.step / timeline.unit;
            }
            return 0;
        }

        function getPhaseX(phase, timeline) {
            if (angular.isDefined(timeline.start) && angular.isDefined(phase.startDate)) {
                var time = phase.startDate.diff(timeline.start, timeline.minimalMetric, true);
                return time * timeline.step / timeline.unit;
            }
            return 0;
        }

        function getPlannedSectionWidth(endDate, timeline) {
            if (angular.isDefined(endDate) && angular.isDefined(timeline.start)) {
                var time = endDate.diff(timeline.start, timeline.minimalMetric, true);
                return Math.ceil(time * timeline.step / timeline.unit);
            }
            return 0;
        }

        function getDelaySectionWidth(endDate, expectedEndDate, timeline) {
            if (expectedEndDate.isAfter(endDate)) {
                var time = expectedEndDate.diff(timeline.start, timeline.minimalMetric, true);
                return Math.ceil(time * timeline.step / timeline.unit);
            }
            return 0;
        }

        function getElapsedSectionWidth(timeline) {
            if (angular.isObject(timeline) && angular.isDefined(timeline.start)) {
                var end = moment();
                var time = end.diff(timeline.start, timeline.minimalMetric, true);
                return time * timeline.step / timeline.unit;
            }
            return 0;
        }

        function createTimeline(container) {

            var timeline = {};
            timeline.minimalMetric = container.minimalMetric;
            timeline.columnWidth = vm.timelineWidth / container.size;
            timeline.step = timeline.columnWidth / container.segments;
            timeline.textIndent = STEP_INDENT;
            timeline.titles = container[container.metric];
            timeline.range = [];

            timeline.unit = container.stepUnit;
            timeline.start = container.timelineStart;
            timeline.end = container.timelineEnd;
            for (var i = 0; i < container.size; i++) {
                timeline.range.push(i);
            }
            return timeline;
        }
    };
    XlrTimelineDirectiveController.$inject = XlrTimelineDirectiveControllerInject;

    var XlrTimelineChartInject = ['$window', '$timeout', 'Events'];
    var XlrTimelineChart = function XlrTimelineChart($window, $timeout, Events) {
        return {
            bindToController: {
                timelineData: '=',
                isDetailsView: '=detailView'
            },
            require: ['xlrTimelineChart', '?^xlrTile'],
            link: function link($scope, elem, attr, ctrls) {
                var ctrl = ctrls[0];
                var xlrTile = ctrls[1];

                var PANEL_BODY_PADDING = 30;

                if (ctrl.isDetailsView) {
                    var resizeFn = function resizeFn() {
                        $timeout(function () {
                            ctrl.redrawTimeline($window.innerWidth - 110);
                        }, 200);
                    };
                    angular.element($window).bind('resize', resizeFn);
                    $scope.$on('$destroy', function () {
                        angular.element($window).unbind('resize', resizeFn);
                    });
                    ctrl.init($window.innerWidth - 110);
                } else {
                    if (xlrTile) {
                        ctrl.init(xlrTile.width, xlrTile.height - PANEL_BODY_PADDING);
                    }
                    $scope.$on(Events.tile.resized, function (event) {
                        if (xlrTile) {
                            ctrl.redrawTimeline(xlrTile.width, xlrTile.height - PANEL_BODY_PADDING);
                        }
                    });
                }
            },
            controller: 'xlrTimelineDirectiveController',
            controllerAs: 'timelineCtrl',
            templateUrl: 'static/5.0.0/include/TimelineTile/partials/timeline-chart.html'
        };
    };
    XlrTimelineChart.$inject = XlrTimelineChartInject;

    angular.module('xlrelease').controller('xlrTimelineDirectiveController', XlrTimelineDirectiveController);
    angular.module('xlrelease').directive('xlrTimelineChart', XlrTimelineChart);
})();