#
# Copyright (c) 2021. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#

import sets
import com.xhaus.jyson.JysonCodec as json

from xlrelease.HttpRequest import HttpRequest

HTTP_SUCCESS = sets.Set([200, 201, 204])

class TeamforgeClient(object):

    def __init__(self, server):
        self.httpParams = server;
        self.tokenUsername = server['username']
        self.tokenPassword = server['password']
        self.httpParams['username'] = None
        self.httpParams['password'] = None
        self.httpParams['authenticationMethod'] = 'None'
        self.httpParams['oauth2GrantType'] = None
        self.httpRequest = HttpRequest(self.httpParams)
        self.tokenHeader = {"Accept": "application/json", "Content-Type": "application/x-www-form-urlencoded"}
        self.scope = 'urn:ctf:services:ctf'
        self.token = ''

    @staticmethod
    def create_teamforge_client(server):
        return TeamforgeClient(server)

    def get_token(self):
        if self.token is not None and self.token != "":
            return self.token
        else:
            url = "/oauth/auth/token"
            body = "grant_type=password&client_id=api-client&scope=" + self.scope + "&username=" + self.tokenUsername + "&password=" + self.tokenPassword
            response = self.httpRequest.post(url, headers=self.tokenHeader, body=body)
            results = self.get_results(response)
            self.token = results['access_token']
            return self.token

    def get_headers(self, token):
        return {"Accept": "application/json", "Content-Type": "application/json", "Authorization": "Bearer %s" % token}

    def http_get_url(self, url):
        token = self.get_token()
        response = self.httpRequest.get(url, headers=self.get_headers(token))
        return self.get_results(response)

    def http_post_url(self, url, body):
        token = self.get_token()
        response = self.httpRequest.post(url, headers=self.get_headers(token), body=body)
        return self.get_results(response)

    def http_put_url(self, url, body):
        token = self.get_token()
        response = self.httpRequest.put(url, headers=self.get_headers(token), body=body)
        return self.get_results(response)

    def http_patch_url(self, url, body):
        token = self.get_token()
        headers = self.get_headers(token)
        headers["If-Match"]= "W/*"
        response = self.httpRequest.patch(url, headers=headers, body=body)
        return self.get_results(response)        

    def get_results(self, response):
        if response.getStatus() in HTTP_SUCCESS:
            if response.getResponse() is not None:
               return json.loads(response.getResponse())
        else:
            raise Exception("Error from server, HTTP Return: %s, content %s" % (response.getStatus(), response.getResponse()))

    def get_project(self, projectId):
        url = '/ctfrest/foundation/v1/projects/'+projectId
        return self.http_get_url(url)

    def create_project(self, body):
        url = '/ctfrest/foundation/v1/projects'
        return self.http_post_url(url, body)

    def create_artifact(self, trackerid, body):
        url = '/ctfrest/tracker/v1/trackers/'+trackerid+'/artifacts'
        return self.http_post_url(url, body)
    
    def get_artifact(self, artfId):
        url = '/ctfrest/tracker/v1/artifacts/'+artfId
        return self.http_get_url(url)

    def getArtifactDetails(self, artifactId):
        url = "/ctfrest/tracker/v1/artifacts/%s" % artifactId

        # make the request get call
        data = self.http_get_url(url)
        return self.parseApiResponse(data) 

    def getArtifactChildren(self, parentArtifactId):
        url = "/ctfrest/tracker/v1/artifacts/%s" % parentArtifactId
        url = url + "/dependencies/children"
        
        # make the request get call
        data = self.http_get_url(url)
        childrenArtifactIds = []
        for key in data:
            if key == "items":
                for subKey in data['items']:
                    childrenArtifactIds += [subKey['id']]
        return childrenArtifactIds

    def getAssociations(self, objectId):
        url = "/ctfrest/foundation/v1/objects/%s" % objectId
        url = url + "/associations?depth=0&includeDependencies=false&includeEvents=true&includePhantoms=false&limit=0"

        # make the request get call
        data = self.http_get_url(url)
        return self.parseApiResponse(data)

    def createAssociations(self, objectIds, targetId):
        for objectId in objectIds:
            url = "/ctfrest/foundation/v1/objects/%s" % objectId
            url = url + "/associations/%s" % targetId

            payload = {}
            body = json.dumps(payload)

            # make the request put call
            data = self.http_put_url(url, body)

        # Now that the association(s) have been made, make a get call for the associations
        return self.getAssociations(targetId)

    def updateArtifact(self, artifactIds, status, comment):
        data = {}
        for artifactId in artifactIds:
            url = "/ctfrest/tracker/v1/artifacts/%s" % artifactId
            payload = {'status': status, 'comment': comment}
            body = json.dumps(payload)

            # make the request patch call
            data = self.http_patch_url(url, body)

        return self.parseApiResponse(data)

    def parseApiResponse(self, data):
        flexFields = {}
        fields = {}
        cleanData = self.cleanNullTerms(data)
        for key in cleanData:
            if key == "flexFields":
                for subKey in cleanData['flexFields']:
                    try:
                        flexFields[subKey['name']] = subKey['values'][0]
                    except:
                        print("Field Exception in flexFields")
            else:
                try:
                    fields[key] = cleanData[key]
                except:
                    print("Field Exception in Fields")
        return fields, flexFields

    def cleanNullTerms(self, d):
        clean = {}
        for k, v in d.items():
            if isinstance(v, dict):
                nested = self.cleanNullTerms(v)
                if len(nested.keys()) > 0:
                    clean[k] = nested
            elif v is not None and v != "":
                clean[k] = v
        return clean
