#
# Copyright 2017 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import sys
import string
import com.xhaus.jyson.JysonCodec as json
from vsts.HttpRequest import HttpRequest
from vsts.util import error

class TfsServer(object):
    """Exceptions are documented in the same way as classes.

    The __init__ method may be documented in either the class level
    docstring, or as a docstring on the __init__ method itself.

    Either form is acceptable, but the two should not be mixed. Choose one
    convention to document the __init__ method and be consistent with it.
    """

    __json_patch = 'application/json-patch+json'
    __json = 'application/json'

    def __init__(self, tfs_server, username = None, password = None, domain = None):
        if tfs_server is None:
            error('No server provided.')

        self.request = HttpRequest(tfs_server, username, password, domain)

    def addWorkItemComment(self, workItemId, workItemComment):
        content = '[%s]' % self._formatField("System.History", workItemComment)
        response = self.request.patch('_apis/wit/workitems/%s?api-version=1.0' % workItemId, content, contentType=self.__json_patch)

        if response.status != 200:
            error("Failed to add comment.", response)

    def addWorkItemTag(self, workItemId, tags):
        if not tags:
            return

        fields = self.getWorkItem(workItemId)

        if fields.has_key("System.Tags"):
            tags.extend(fields["System.Tags"].split(';'))

        self.updateWorkItemField(workItemId, "System.Tags", ";".join(tags))

    def createWorkItem(self, teamProjectName, workItemType, workItemTitle, fields):
        if not teamProjectName:
            raise Exception("Team project name is a mandatory field.")

        if not workItemType:
            raise Exception("Work Item Type is a mandatory field.")

        if not workItemTitle:
            raise Exception("Work Item Title is a mandatory field.")

        items = [self._formatField("System.Title", workItemTitle)]

        for key, value in fields.iteritems():
            items.append(self._formatField(key, value))

        response = self.request.patch('%s/_apis/wit/workitems/$%s?api-version=1.0' % (teamProjectName, workItemType), \
                                       self._formatJsonContent(items), \
                                       contentType=self.__json_patch)

        if response.status == 200:
            return json.loads(response.response)['id']
        else:
            error("Failed to create work item.", response)

    def getWorkItem(self, workItemId):
        if not workItemId:
            raise Exception("Work Item Id is a mandatory field.")

        response = self.request.get('_apis/wit/workitems/%s?api-version=1.0' % workItemId)

        if not response.isSuccessful():
            error("Error fetching work item.", response)

        return json.loads(response.response)['fields']

    def getWorkItemState(self, workItemId):
        return self.getWorkItem(workItemId)["System.State"]

    def wiqlWorkItem(self, query):
        content = '{ "query": %s }' %  json.dumps(query)

        response = self.request.post('_apis/wit/wiql?api-version=1.0', content, contentType=self.__json)

        if response.status != 200:
            error("Failed to execute the query.", response)

        data = json.loads(response.response)

        ids = []
        for item in data['workItems']:
            ids.append(item['id'])

        return ids

    def queryWorkItem(self, teamProjectName, queryName):
        if not teamProjectName:
            raise Exception("Team project name is a mandatory field.")

        if not queryName:
            raise Exception("Query Name is a mandatory field.")

        response = self.request.get('%s/_apis/wit/queries/%s' % (teamProjectName, queryName))

        if response.status != 200:
            error("Query with name '%s' not found." % queryName, response)


        query = json.loads(response.response)
        link = query['_links']['wiql']['href']
        
        response = self.request.get(link)

        if response.status != 200:
            error("Failed to execute the query.", response)

        data = json.loads(response.response)

        ids = []
        for item in data['workItems']:
            ids.append(item['id'])

        return ids

    def getGitItem(self, teamProjectName, repositoryName, scopePath):
        response = self.request.get('%s/_apis/git/repositories/%s/items?api-version=1.0&scopepath=%s' % (teamProjectName, repositoryName, scopePath))

        if response.status != 200:
            error("Error fetching Git item.", response)

        return response.response

    def getTfvcItem(self, path):
        response = self.request.get('_apis/tfvc/items/%s?api-version=1.0' % path)

        if response.status != 200:
            error("Error fetching TFVC item.", response)

        return response.response

    def getBuildDefinitionId(self, teamProjectName, buildDefinitionName):
        if not teamProjectName:
            raise Exception("Team project name is a mandatory field.")

        if not buildDefinitionName:
            raise Exception("Build Definition Name is a mandatory field.")

        response = self.request.get('%s/_apis/build/definitions?api-version=2.0&name=%s' % (teamProjectName, buildDefinitionName))

        if response.status != 200:
            error("Error fetching build definition.", response)
        
        buildDefinition = json.loads(response.response)

        if buildDefinition["count"] == 0:
            error("Build definition with name %s was not found." % buildDefinitionName)

        return buildDefinition["value"][0]["id"]

    def queueBuild(self, teamProjectName, buildDefinitionId, buildProperties):
        params = '"parameters": "{'
        mapSize = len(buildProperties)

        for buildProperty in buildProperties.keys():
            key = buildProperty
            value = buildProperties.get(buildProperty)
            params += "'" + key + "'" + ":"
            params += "'" + value + "'"
            mapSize -= 1
            if mapSize > 0:
                params += ","
            else:
                params += '}"'

        content = '{"definition": { "id": %s }, %s }' % (buildDefinitionId, params)
        response = self.request.post('%s/_apis/build/builds?api-version=2.0' % teamProjectName, content, contentType=self.__json)

        if response.status != 200:
            error("Failed to queue build.", response)

        return json.loads(response.response)["id"]

    def getBuild(self, teamProjectName, buildId):
        response = self.request.get('%s/_apis/build/builds/%s?api-version=2.0' % (teamProjectName, buildId))

        if response.status != 200:
            error("Failed to fetch build status.", response)

        return json.loads(response.response)

    def getBuildStatus(self, teamProjectName, buildId):
        return self.getBuild(teamProjectName, buildId)["status"]

    def getBuildNumber(self, teamProjectName, buildId):
        return self.getBuild(teamProjectName, buildId)["buildNumber"]

    def getBuildResult(self, teamProjectName, buildId):
        return self.getBuild(teamProjectName, buildId)["result"]

    def addBuildTag(self, teamProjectName, buildId, tag):
        if not teamProjectName:
            raise Exception("Team project name is a mandatory field.")

        if not buildId:
            raise Exception("Build Id is a mandatory field.")

        tag = json.dumps(tag).strip('"')
        response = self.request.put('%s/_apis/build/builds/%s/tags/%s?api-version=2.0' % (teamProjectName, buildId, tag), "", contentType=self.__json)

        if response.status != 200:
            error("Failed to add build tag.", response)

    def updateWorkItemField(self, workItemId, fieldName, fieldValue):
        self.updateWorkItemFields(workItemId, {fieldName:fieldValue})

    def updateWorkItemFields(self, workItemId, fields):
        items = []
        for key, value in fields.iteritems():
            items.append(self._formatField(key, value))

        response = self.request.patch('_apis/wit/workitems/%s?api-version=1.0' % workItemId, self._formatJsonContent(items), contentType=self.__json_patch)
        
        if response.status != 200:
            error("Failed to update field(s).", response)

    def getChangesets(self, teamProjectName):
        response = self.request.get('%s/_apis/tfvc/changesets?api-version=1.0' % teamProjectName)

        if response.status != 200:
            error("Failed to fetch changesets.", response)

        changeSets = json.loads(response.response)

        if changeSets["count"] > 0:
            return changeSets["value"]
    
    def getCommits(self, teamProjectName, repositoryName, branchName):
        response = self.request.get('%s/_apis/git/repositories/%s/commits?api-version=1.0&branch=%s&$top=2' % (teamProjectName, repositoryName, branchName))

        if response.status != 200:
            error("Failed to fetch commits.", response)

        commits = json.loads(response.response)

        if commits["count"] > 0:
            return commits["value"]

    def _formatField(self, fieldName, fieldValue):
        if not fieldName or not fieldValue:
            raise Exception("Field name or value are not specified.")

        return'{"path": "/fields/%s", "value": %s, "op": "add"}' % (fieldName, json.dumps(fieldValue))

    def _formatJsonContent(self, items):
        if items:
            return ",".join(items).join(("[","]"))
