import React, { MouseEvent, useState, useEffect } from 'react';
import ReactMarkdown from 'react-markdown';
import isString from 'lodash/isString';
import remarkGfm from 'remark-gfm';
import classNames from 'classnames';
import { DotButton, DotIconButton, DotInputText, DotTypography } from '@digital-ai/dot-components';
import { HelpButton } from '../../../features/main-navigation/help-button/help-button';
import { mentionsTrigger, OptionSelectorInput } from '../../../features/common/components/options-selector/option-selector-input.component';
import { ReleaseVariable, User } from '../../../types';
import getAngularService from '../../../features/common/services/angular-accessor';
import { VariablesInterpolatorFactory } from '../../../features/tasks/types/angular';
import { mentionsRegex } from '../../../js/util/filters';
import './textarea-markdown.component.less';

export interface TextareaMarkdownProps {
    enableMentions?: boolean;
    inputId: string;
    isReadOnly?: boolean;
    isValueRequired?: boolean;
    maxRows?: number;
    name: string;
    onDelete?: (event: MouseEvent<HTMLButtonElement>) => void;
    onTextChange: (text: string) => void;
    placeholder?: string;
    rows: number;
    text: string;
    users?: User[];
    variables?: ReleaseVariable;
}

export const TextAreaMarkdown = ({
    enableMentions = false,
    inputId,
    isReadOnly = false,
    isValueRequired = true,
    maxRows,
    name,
    onDelete,
    onTextChange,
    placeholder,
    rows,
    text,
    users = [],
    variables,
}: TextareaMarkdownProps) => {
    const [editing, setEditing] = useState<boolean>(false);
    const [originalText, setOriginalText] = useState<string>(text);
    const [currentText, setCurrentText] = useState<string>(text);
    const variablesInterpolator = getAngularService('VariablesInterpolator') as never as VariablesInterpolatorFactory;

    useEffect(() => {
        setOriginalText(text);
        setCurrentText(text);
    }, [text]);

    const saveChanges = () => {
        const trimmedText = currentText.trim();
        if (isValueRequired && !trimmedText) {
            revertChanges();
            return;
        }
        onTextChange(trimmedText);
        setOriginalText(trimmedText);
        setEditing(false);
    };

    const revertChanges = () => {
        setCurrentText(originalText);
        setEditing(false);
    };

    const handleDelete = (event: MouseEvent<HTMLButtonElement>) => {
        onDelete && onDelete(event);
        setEditing(false);
    };

    const renderTextEditor = () => (
        <div>
            {enableMentions ? (
                <OptionSelectorInput
                    id={inputId}
                    maxRows={maxRows}
                    multiline={true}
                    name={name}
                    onChange={(inputValue) => setCurrentText(inputValue)}
                    optionsList={users}
                    rows={rows}
                    trigger={mentionsTrigger}
                    value={currentText}
                />
            ) : (
                <DotInputText
                    id={inputId}
                    multiline={true}
                    name={name}
                    onChange={(e) => setCurrentText(e.target.value)}
                    rows={rows}
                    rowsMax={maxRows}
                    value={currentText}
                />
            )}
            <div className="actions-wrapper">
                <DotButton data-testid="cancel-button" onClick={revertChanges} type="text">
                    Cancel
                </DotButton>
                <DotButton data-testid="save-button" onClick={saveChanges} type="primary">
                    Save
                </DotButton>
                <HelpButton helpLink="how-to/using-markdown-in-xl-release.html" />
            </div>
        </div>
    );

    const wrapMentionsText = (textWithMention: string) => {
        return textWithMention.replace(mentionsRegex, (match, username) => {
            const prefix = match.substring(0, match.indexOf(mentionsTrigger));
            return `${prefix}**${mentionsTrigger}${username}**`;
        });
    };

    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    const renderMarkDownStrongElement = ({ children, ...props }) => {
        if (enableMentions && children && children.length === 1 && isString(children[0]) && children[0].startsWith(mentionsTrigger)) {
            const user = users.find((currentUser) => currentUser.username === children[0].substring(mentionsTrigger.length));
            if (user) {
                return (
                    <span className="mention" title={user.username}>
                        {user.fullName || user.username}
                    </span>
                );
            }
        }
        return <strong {...props}>{children}</strong>;
    };

    const replaceVariables = (value: string) => {
        if (variables && value) {
            return variablesInterpolator.interpolateInText(variables, value);
        } else {
            return value;
        }
    };

    const getRemarkPlugins = (markdownText: string) => {
        // https://github.com/micromark/micromark-extension-gfm-table/issues/6
        // there is performance issue when number of lines are bigger than X
        // testing on my laptop this was an okeish performance
        const lines = (markdownText.match(/\n/g) || '').length + 1;
        return lines > 300 ? [] : [remarkGfm];
    };

    const mouseUp = (e: MouseEvent<HTMLDivElement | HTMLAnchorElement>) => {
        const maybeIcon = (e.target as HTMLDivElement).classList.contains('dot-i');
        const maybeHref = (e.target as HTMLAnchorElement).href;
        // if we are in read only mode, don't do anything
        // if we are clicking on some action icon, don't do anything
        // if we are clicking on a href from markdown, don't do anything
        if (isReadOnly || maybeIcon || maybeHref) {
            return;
        }

        const maybeSelectedText = window.getSelection()?.toString();
        // if we didn't select text, edit mode goes brrrr
        if (!maybeSelectedText) {
            setEditing(true);
        }
    };

    const renderMarkDownPreview = () => (
        <div className={classNames('markdown-viewer', { 'read-only': isReadOnly })} onMouseUp={mouseUp} role="button" tabIndex={isReadOnly ? undefined : 0}>
            {!isReadOnly ? (
                <div className="markdown-viewer-actions">
                    {onDelete && originalText && <DotIconButton data-testid="delete-button" iconId="delete" onClick={handleDelete} size="small" />}
                    <DotIconButton data-testid="edit-button" iconId="edit" onClick={() => setEditing(true)} size="small" />
                </div>
            ) : null}
            {currentText ? (
                <ReactMarkdown className="markdown-wrapper" components={{ strong: renderMarkDownStrongElement }} remarkPlugins={getRemarkPlugins(currentText)}>
                    {enableMentions ? replaceVariables(wrapMentionsText(currentText)) : replaceVariables(currentText)}
                </ReactMarkdown>
            ) : (
                !isReadOnly && (
                    <div className="placeholder-wrapper" onClick={() => setEditing(true)}>
                        <DotTypography className="placeholder" variant="body1">
                            {placeholder}
                        </DotTypography>
                    </div>
                )
            )}
        </div>
    );

    return <div className="markdown-switcher">{editing ? renderTextEditor() : renderMarkDownPreview()}</div>;
};
