import React from 'react';
import { ReactWrapper } from 'enzyme';
import { mountWithStoreAndTheme } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ENDPOINT_TYPE, SERVER_AUTHENTICATION_METHOD, SERVER_AUTHENTICATION_OPTIONS } from '../../../constants';
import { AutoCompleteValue, DotAlertBanner, DotAutoComplete, DotButton, DotInputText } from '@digital-ai/dot-components';
import { folderExternalDeployments, initialState } from '../../../ducks/external-deployments.reducer';
import { externalApplicationWizard, initialState as wizardInitialState } from '../../../ducks/application-wizard.reducer';
import { ApplicationCreateServerFormProp, CreateServerForm } from './application-create-server-component';
import { Server } from '../../../external-deployment.types';

const dispatch = jest.fn();
const { createServer } = folderExternalDeployments.actions;
const { setConnectionTested, testConnection } = externalApplicationWizard.actions;
describe('Create server component', () => {
    let wrapper: ReactWrapper;
    const closeForm = jest.fn();
    const defaultProps = {
        closeForm,
        folder: { id: 'folderId', title: 'Folder-1' },
        serverType: ENDPOINT_TYPE.deploy,
    };

    const mountComponent = (wizardState = wizardInitialState, props: ApplicationCreateServerFormProp = defaultProps) => {
        const state = {
            externalApplicationWizard: wizardState,
            folderExternalDeployments: initialState,
        };
        wrapper = mountWithStoreAndTheme(<CreateServerForm {...props} />, dispatch, state);
    };

    const server: Server = {
        id: null,
        authenticationMethod: SERVER_AUTHENTICATION_OPTIONS[0].title,
        folderId: 'folderId',
        title: '',
        type: ENDPOINT_TYPE.deploy,
        url: '',
    };

    const doChangeAutocomplete = (value: AutoCompleteValue | null) => {
        wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, value, '');
    };

    const doChangeInput = (index: number, value: string | number) => {
        wrapper.find(DotInputText).at(index).invoke('onChange')?.({
            target: {
                value,
            },
        } as never);
    };

    beforeEach(() => {
        mountComponent();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    it('should contain title', () => {
        expect(wrapper.find(`h1`)).toExist();
        expect(wrapper.find(`h1`).text()).toStrictEqual('New Deploy server');
    });

    it('should contain input for server title', () => {
        expect(wrapper.find(DotInputText).at(0).props().label).toStrictEqual('Title');
    });

    it('should contain input for server url', () => {
        expect(wrapper.find(DotInputText).at(1).props().label).toStrictEqual('URL');
    });

    it('should contain autocomplete for server authentication', () => {
        expect(wrapper.find(DotAutoComplete).exists).toBeTruthy();
        expect(wrapper.find(DotAutoComplete).props().options?.length).toEqual(4);
    });

    it('should unset auth method', () => {
        doChangeAutocomplete(null);
        expect(wrapper.find(DotInputText).at(2).props().label).toStrictEqual('Username');
        expect(wrapper.find(DotInputText).at(3).props().label).toStrictEqual('Password');
    });

    it('should generate input field for basic auth', () => {
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[1]);
        expect(wrapper.find(DotInputText).at(2).props().label).toStrictEqual('Username');
        expect(wrapper.find(DotInputText).at(3).props().label).toStrictEqual('Password');
    });

    it('should generate input field for ntlm auth', () => {
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[2]);
        expect(wrapper.find(DotInputText).at(2).props().label).toStrictEqual('Username');
        doChangeInput(2, 'username');
        expect(wrapper.find(DotInputText).at(3).props().label).toStrictEqual('Password');
        doChangeInput(3, 'password');
        expect(wrapper.find(DotInputText).at(4).props().label).toStrictEqual('Domain');
        doChangeInput(4, 'Domain');
        wrapper.find(`button[data-testid="save-server-btn"]`).simulate('submit');
        expect(dispatch).toBeCalledWith(
            createServer({
                ...server,
                authenticationMethod: SERVER_AUTHENTICATION_OPTIONS[2].id,
                domain: 'Domain',
                password: 'password',
                username: 'username',
            }),
        );
    });

    it('should generate input field for oauth2 auth', () => {
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[3]);
        expect(wrapper.find(DotInputText).at(2).props().label).toStrictEqual('Username');
        expect(wrapper.find(DotInputText).at(3).props().label).toStrictEqual('Password');
        expect(wrapper.find(DotInputText).at(4).props().label).toStrictEqual('Access Token URL');
        doChangeInput(4, 'token');
        expect(wrapper.find(DotInputText).at(5).props().label).toStrictEqual('Client ID');
        doChangeInput(5, 'clientId');
        expect(wrapper.find(DotInputText).at(6).props().label).toStrictEqual('Client Secret');
        doChangeInput(6, 'clientSecret');
        expect(wrapper.find(DotInputText).at(7).props().label).toStrictEqual('Scope');
        doChangeInput(7, 'scope');
        wrapper.find(`button[data-testid="save-server-btn"]`).simulate('submit');
        expect(dispatch).toBeCalledWith(
            createServer({
                ...server,
                authenticationMethod: SERVER_AUTHENTICATION_OPTIONS[3].id,
                accessTokenUrl: 'token',
                clientId: 'clientId',
                clientSecret: 'clientSecret',
                scope: 'scope',
            }),
        );
    });

    it('should contain proxy fields', () => {
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[0]);
        expect(wrapper.find(DotInputText).at(2).props().label).toStrictEqual('Proxy Host');
        doChangeInput(2, 'host');
        expect(wrapper.find(DotInputText).at(3).props().label).toStrictEqual('Proxy Port');
        doChangeInput(3, 8082);
        expect(wrapper.find(DotInputText).at(4).props().label).toStrictEqual('Proxy Username');
        doChangeInput(4, 'username');
        expect(wrapper.find(DotInputText).at(5).props().label).toStrictEqual('Proxy Password');
        doChangeInput(5, 'pass');
        expect(wrapper.find(DotInputText).at(6).props().label).toStrictEqual('Proxy Domain');
        doChangeInput(6, 'domain');
        wrapper.find(`button[data-testid="save-server-btn"]`).simulate('submit');
        expect(dispatch).toBeCalledWith(
            createServer({
                ...server,
                authenticationMethod: SERVER_AUTHENTICATION_OPTIONS[0].id,
                proxyDomain: 'domain',
                proxyHost: 'host',
                proxyPort: 8082,
                proxyPassword: 'pass',
                proxyUsername: 'username',
            }),
        );
    });

    it("should display disabled 'Save' button", () => {
        expect(wrapper.find(`button[data-testid="save-server-btn"]`).exists).toBeTruthy();
        expect(wrapper.find(`button[data-testid="save-server-btn"]`).text()).toStrictEqual('Save');
        expect(wrapper.find(`button[data-testid="save-server-btn"]`).props().disabled).toStrictEqual(true);
    });

    it("should display enabled 'Save' button", () => {
        wrapper.find(DotInputText).at(0).invoke('onBlur')?.('' as never);
        wrapper.find(DotInputText).at(1).invoke('onBlur')?.('' as never);
        wrapper.update();
        expect(wrapper.find(DotInputText).at(0).props().helperText).toStrictEqual('Title is mandatory');
        expect(wrapper.find(DotInputText).at(1).props().helperText).toStrictEqual('Url is mandatory');
        doChangeInput(0, 'test-title');
        doChangeInput(1, 'test-url');
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[0]);
        expect(wrapper.find(`button[data-testid="save-server-btn"]`).props().disabled).toStrictEqual(false);
    });

    it("should handle 'Save' button click", () => {
        doChangeInput(0, 'test-title');
        doChangeInput(1, 'test-url');
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[0]);
        wrapper.find(`button[data-testid="save-server-btn"]`).simulate('submit');
        expect(dispatch).toBeCalledWith(createServer({ ...server, title: 'test-title', url: 'test-url' }));
    });

    it("should handle 'Test' button click", () => {
        doChangeInput(0, 'test-title');
        doChangeInput(1, 'test-url');
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[0]);
        wrapper.find(`button[data-testid="test-connection-btn"]`).simulate('click');
        expect(dispatch).toBeCalledWith(
            testConnection({
                id: null,
                properties: {
                    authenticationMethod: SERVER_AUTHENTICATION_METHOD.none,
                    folderId: defaultProps.folder.id,
                    id: null,
                    title: 'test-title',
                    type: defaultProps.serverType,
                    url: 'test-url',
                },
                title: 'test-title',
                type: defaultProps.serverType,
            }),
        );
    });

    it('should render alert banner', () => {
        mountComponent({ ...wizardInitialState, connectionTested: true, testConnectionStatus: { errorText: '', success: true } });
        const alertBanner = wrapper.find(DotAlertBanner);
        expect(alertBanner).toExist();
        expect(alertBanner.props().severity).toStrictEqual('success');
        expect(wrapper.find('.create-server-test-message').text()).toStrictEqual('Digital.ai Deploy Server is available.');
    });

    it('should render error alert banner', () => {
        mountComponent({ ...wizardInitialState, connectionTested: true, testConnectionStatus: { errorText: 'errorText', success: false } });
        const alertBanner = wrapper.find(DotAlertBanner);
        expect(alertBanner.props().severity).toStrictEqual('error');
        expect(wrapper.find('.create-server-test-message').text()).toStrictEqual("Can't connect to Digital.ai Deploy Server.errorText");
    });

    it('should render error alert banner', () => {
        mountComponent(
            { ...wizardInitialState, connectionTested: true, testConnectionStatus: { errorText: '', success: true } },
            { ...defaultProps, serverType: ENDPOINT_TYPE.argoCd },
        );
        const alertBanner = wrapper.find(DotAlertBanner);
        expect(alertBanner).toExist();
        expect(alertBanner.props().severity).toStrictEqual('success');
        expect(wrapper.find('.create-server-test-message').text()).toStrictEqual('ArgoCD Server is available.');
        wrapper.find(DotInputText).at(4).invoke('onChange')?.({ target: { value: 'auth' } } as never);
        wrapper.find('button').at(0).simulate('click');
        wrapper.update();
        expect(dispatch).toHaveBeenCalledWith(setConnectionTested(false));
    });
});
