import React from 'react';
import { ReactWrapper } from 'enzyme';
import { DotAutoComplete, DotCardHeader } from '@digital-ai/dot-components';
import noop from 'lodash/noop';
import { mountWithStoreAndTheme } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { FilterApplications, FilterApplicationsProps } from './filter-applications-component';
import { argoServerMock, deployServerMock, mockDeployWebhookSource } from '../../../__mocks__/external-deployments.mocks';
import { folderExternalDeployments, FolderExternalDeploymentsState, initialState } from '../../../ducks/external-deployments.reducer';
import { getFilterCardTitle, getServerIcon } from '../../../helper/utils';
import { IMG_SRC } from '../../../constants';
import { PatchModal } from '../patch-modal/patch-modal-component';
import { EventSourceForm } from '../event-source/application-event-source.component';

const { loadWebhookSourceFilters, updateWebhookSourceFilters, patchExternalDeployments } = folderExternalDeployments.actions;
const dispatch = jest.fn();

describe('FilterApplications component', () => {
    let wrapper: ReactWrapper;
    const onSave = jest.fn();
    const defaultProps = {
        onCancel: () => noop(),
        onFilterBack: () => noop(),
        onSave,
        server: deployServerMock,
        webhookSource: mockDeployWebhookSource,
    };
    const defaultState = {
        ...initialState,
    };
    const mountComponent = (state: FolderExternalDeploymentsState = defaultState, props: FilterApplicationsProps = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<FilterApplications {...props} />, dispatch, { folderExternalDeployments: state });
        expect(dispatch).toBeCalledWith(loadWebhookSourceFilters(mockDeployWebhookSource.id || ''));
    };

    beforeEach(() => {
        mountComponent();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    const searchDotCard = (selector: string) => wrapper.find(`.filter-applications ${selector}`);

    it('should show the header', () => {
        expect(searchDotCard('.dot-typography').at(0).text()).toStrictEqual('Map folders');
    });

    it('should show folder filters autocomplete', () => {
        expect(searchDotCard('.folders-filter').at(0).text()).toContain('Choose folders');
        expect(searchDotCard('.folders-filter').at(0).text()).toContain('Map folders to display applications from');
    });

    it('should show server card', () => {
        expect(searchDotCard('.filter-card').at(0).text()).toContain(getFilterCardTitle(deployServerMock.type || ''));
        expect(searchDotCard('.filter-card').at(0).text()).toContain(deployServerMock.url);
        expect(searchDotCard('.filter-card img').at(0).props().src).toStrictEqual(getServerIcon(deployServerMock.type || ''));
    });

    it('should show webhook source card', () => {
        expect(searchDotCard('.filter-card').at(1).text()).toContain(mockDeployWebhookSource.title);
        expect(searchDotCard('.filter-card img').at(1).props().src).toStrictEqual(IMG_SRC.webhookSource);
    });

    it('should show buttons', () => {
        expect(searchDotCard('.dot-button button').at(0).text()).toStrictEqual('Back');
        expect(searchDotCard('.dot-button button').at(1).text()).toStrictEqual('Cancel');
        expect(searchDotCard('.dot-button button').at(2).text()).toStrictEqual('Done');
    });

    it('should save deploy filters and update patch modal manually', () => {
        mountComponent({
            ...defaultState,
            webhookSourceFilters: {
                folderFilterOptions: [],
                folderFilterValues: ['folder'],
                webhookSourceId: 'some',
            },
        });
        searchDotCard(`button[data-testid="filter-apps-done-btn"]`).simulate('click');
        wrapper.update();
        expect(dispatch).toHaveBeenCalledWith(
            updateWebhookSourceFilters({
                filteredFolders: ['folder'],
                webhookSourceId: defaultProps.webhookSource.id || '',
            }),
        );
        expect(onSave).toHaveBeenCalled();
        const patchModal = wrapper.find(PatchModal);
        patchModal.invoke('onModalUpdateManually')?.();
        wrapper.update();
        expect(wrapper.find(PatchModal).props().isModalOpen).toBeFalsy();
        expect(onSave).toHaveBeenCalled();
    });

    it('should save argo filters and update patch modal automatically', () => {
        mountComponent(
            {
                ...defaultState,
                webhookSourceFilters: {
                    folderFilterOptions: [],
                    folderFilterValues: ['folder'],
                    webhookSourceId: 'some',
                },
            },
            { ...defaultProps, server: argoServerMock },
        );
        searchDotCard(`button[data-testid="filter-apps-done-btn"]`).simulate('click');
        wrapper.update();
        expect(dispatch).toHaveBeenCalledWith(
            updateWebhookSourceFilters({
                filteredFolders: ['folder'],
                webhookSourceId: defaultProps.webhookSource.id || '',
            }),
        );
        const patchModal = wrapper.find(PatchModal);
        expect(patchModal.props().isModalOpen).toBeTruthy();
        patchModal.invoke('onModalUpdateAutomatically')?.();
        wrapper.update();
        expect(dispatch).toHaveBeenCalledWith(patchExternalDeployments(defaultProps.webhookSource.id || ''));
        expect(wrapper.find(PatchModal).props().isModalOpen).toBeFalsy();
        expect(onSave).toHaveBeenCalled();
    });

    it('should choose folder', () => {
        mountComponent({
            ...defaultState,
            webhookSourceFilters: {
                folderFilterOptions: ['folder'],
                folderFilterValues: [],
                webhookSourceId: 'some',
            },
        });
        const autocomplete = wrapper.find(DotAutoComplete);
        autocomplete.invoke('onChange')?.('' as never, [{ title: 'folder' }], '');
        wrapper.update();
        expect(wrapper.find(DotAutoComplete).props().value).toStrictEqual([{ id: 'folder', title: 'folder' }]);
    });

    it('should edit webhook source form', () => {
        searchDotCard(`button[data-testid="filter-webhook-edit-action"]`).simulate('click');
        wrapper.update();
        const eventSourceForm = wrapper.find(EventSourceForm);
        expect(eventSourceForm).toExist();
        expect(eventSourceForm.props().eventSourceId).toStrictEqual(defaultProps.webhookSource.eventSource);
        eventSourceForm.invoke('closeForm')?.();
        wrapper.update();
        expect(wrapper.find(EventSourceForm)).not.toExist();
    });

    it('handle undefined server and webhookSource', () => {
        mountComponent(defaultState, {
            ...defaultProps,
            server: undefined,
            webhookSource: { ...mockDeployWebhookSource, id: null, eventSource: null },
        });
        expect(wrapper.find(DotCardHeader).at(0).props().title).toStrictEqual('');
    });
});
