import { SagaIterator } from 'redux-saga';
import { all, call, put, takeLatest } from 'redux-saga/effects';
import { deliveryTasksSaga, loadDeliveryOrPatternSaga, searchDeliveriesOrPatternsSaga, searchPatternsSaga } from './delivery-tasks.saga';
import { fetchDeliveries, getDelivery } from '../../../../deliveries/services';
import { fetchDeliveryPattern, searchDeliveryPatterns } from '../../../../delivery-patterns/delivery-pattern.service';
import { deliveryTasks } from './delivery-tasks.reducer';

const { loadDeliveryOrPattern, setDeliveryOrPattern, searchDeliveriesOrPatterns, setDeliveriesOrPatterns, searchPatterns, setPatterns } = deliveryTasks.actions;

describe('deliveryTasksSaga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = deliveryTasksSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(loadDeliveryOrPattern, loadDeliveryOrPatternSaga),
                takeLatest(searchDeliveriesOrPatterns, searchDeliveriesOrPatternsSaga),
                takeLatest(searchPatterns, searchPatternsSaga),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('loadDeliveryOrPatternSaga', () => {
        it('should call proper saga when it is a delivery', () => {
            const gen: SagaIterator = loadDeliveryOrPatternSaga({ type: '', payload: { isDelivery: true, id: 'DeliveryId' } });
            expect(gen.next().value).toStrictEqual(call(getDelivery, 'DeliveryId', true));
            expect(gen.next({ data: { id: 'DeliveryId', title: 'My delivery' } }).value).toStrictEqual(
                put(
                    setDeliveryOrPattern({
                        id: 'DeliveryId',
                        title: 'My delivery',
                    } as never),
                ),
            );
            expect(gen.next().done).toBe(true);
        });

        it('should call proper saga when it is a pattern', () => {
            const gen: SagaIterator = loadDeliveryOrPatternSaga({ type: '', payload: { isDelivery: false, id: 'PatternId' } });
            expect(gen.next().value).toStrictEqual(call(fetchDeliveryPattern, 'PatternId'));
            expect(gen.next({ id: 'PatternId', title: 'My pattern' }).value).toStrictEqual(
                put(
                    setDeliveryOrPattern({
                        id: 'PatternId',
                        title: 'My pattern',
                    } as never),
                ),
            );
            expect(gen.next().done).toBe(true);
        });

        it('should set the pattern as NOT_FOUND when there is error', () => {
            const gen: SagaIterator = loadDeliveryOrPatternSaga({ type: '', payload: { isDelivery: true, id: 'DeliveryId' } });
            expect(gen.next().value).toStrictEqual(call(getDelivery, 'DeliveryId', true));
            expect(gen.throw?.(new Error('')).value).toStrictEqual(put(setDeliveryOrPattern({ id: 'NOT_FOUND', title: '', stages: [] })));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('searchDeliveriesOrPatternsSaga', () => {
        it('should call proper saga when it is a delivery', () => {
            const gen: SagaIterator = searchDeliveriesOrPatternsSaga({ type: '', payload: { isDelivery: true } });
            expect(gen.next().value).toStrictEqual(
                call(
                    fetchDeliveries,
                    {
                        inProgress: true,
                        title: '',
                    },
                    0,
                    100,
                ),
            );
            expect(gen.next({ data: [{ id: 'DeliveryId', title: 'My delivery' }] }).value).toStrictEqual(
                put(setDeliveriesOrPatterns([{ id: 'DeliveryId', title: 'My delivery' }] as never)),
            );
            expect(gen.next().done).toBe(true);
        });

        it('should call proper saga when it is a pattern', () => {
            const gen: SagaIterator = searchDeliveriesOrPatternsSaga({ type: '', payload: { isDelivery: false } });
            expect(gen.next().value).toStrictEqual(call(searchDeliveryPatterns, { title: '' }));
            expect(gen.next([{ id: 'PatternId', title: 'My pattern' }]).value).toStrictEqual(
                put(
                    setDeliveriesOrPatterns([
                        {
                            id: 'PatternId',
                            title: 'My pattern',
                        },
                    ] as never),
                ),
            );
            expect(gen.next().done).toBe(true);
        });
    });

    describe('searchPatternsSaga', () => {
        it('should call proper saga ', () => {
            const gen: SagaIterator = searchPatternsSaga();
            expect(gen.next().value).toStrictEqual(call(searchDeliveryPatterns, { title: '' }));
            expect(gen.next([{ id: 'PatternId', title: 'My pattern' }]).value).toStrictEqual(
                put(
                    setPatterns([
                        {
                            id: 'PatternId',
                            title: 'My pattern',
                        },
                    ] as never),
                ),
            );
            expect(gen.next().done).toBe(true);
        });
    });
});
