import React from 'react';
import '@testing-library/jest-dom';
import {
    mockIntersectionObserver,
    mockIntersectionObserverWithIntersectingFlag,
    renderWithStoreAndTheme,
    screen,
    userEvent,
    waitFor,
} from '@xlr-ui/tests/unit/testing-utils';
import { releaseMock } from '@xlr-ui/app/features/tasks/__mocks__/release.mock';
import { DrawerHistory, DrawerHistoryProps } from './drawer-history.component';
import { taskMock } from '@xlr-ui/app/features/tasks/__mocks__/task.mock';
import { DEFAULT_FILTER_SETTINGS } from '../../../constants';
import { drawerHistory } from '../ducks/drawer-history.reducer';
import IdsFactory from '@xlr-ui/app/js/util/ids';
import * as helpers from '@xlr-ui/app/features/common/helper';
import { formatDate } from '@xlr-ui/app/features/common/helper';
import moment from 'moment';
import { mockGroupActivityLogs, mockReleaseActivityLogs } from '../../../__mocks__/index.mock';
import { getActivityTypeLabel } from '../../../helper';
import { DRAWER_HISTORY_NUMBER_OF_SKELETONS } from '../constants';

const Ids = IdsFactory();
const formatDateSpy = jest.spyOn(helpers, 'formatDate');

describe('DrawerHistory', () => {
    const release = releaseMock;
    const task = taskMock;
    const taskId = task.id;
    const dispatch = jest.fn();
    const containerId = releaseMock.id;
    const { init, loadFilteredAndPagedHistoryLogs } = drawerHistory.actions;
    const initialFilterSettings = {
        ...DEFAULT_FILTER_SETTINGS,
        targetId: Ids.toDomainId(task.id),
        filter: '',
        dateAsc: false,
    };
    const activityLogs = mockReleaseActivityLogs;
    const dateFormat = 'MM/DD/YYYY';

    const defaultProps: DrawerHistoryProps = {
        expanded: false,
        release,
        task,
    };

    const defaultState = {
        drawerHistory: {
            logs: activityLogs,
            isLastPage: false,
            isLoading: false,
            logsFilter: {
                containerId,
                filterSettings: { ...initialFilterSettings },
                taskId,
            },
            page: 0,
        },
    };

    const getSearch = () => screen.getByPlaceholderText('Filter by user or activity');
    const getOldestFirstBtn = () => screen.queryByRole('button', { name: /Oldest first/i });
    const getNewestFirstBtn = () => screen.queryByRole('button', { name: /Newest first/i });

    const renderComponent = (props = defaultProps, state = defaultState) => {
        renderWithStoreAndTheme(<DrawerHistory {...props} />, dispatch, state);
    };

    beforeEach(() => {
        mockIntersectionObserver();
        formatDateSpy.mockImplementation((date) => moment(date).format(dateFormat));
    });

    afterEach(() => jest.resetAllMocks());

    it('should render header input and sort button', () => {
        renderComponent();
        expect(screen.getByText('History')).toBeInTheDocument();
        expect(getSearch()).toBeInTheDocument();
        expect(screen.getByRole('button', { name: /Newest first/i })).toBeInTheDocument();
    });

    it('should dispatch init action on mount', () => {
        renderComponent();

        expect(dispatch).toHaveBeenCalledWith(
            init({
                containerId: release.id,
                taskId,
                filterSettings: {
                    ...DEFAULT_FILTER_SETTINGS,
                    targetId: Ids.toDomainId(task.id),
                    filter: '',
                    dateAsc: false,
                },
            }),
        );
    });

    it('should dispatch log load action on input change', async () => {
        renderComponent();
        const searchInput = getSearch();
        const filterValue = 'test';

        await userEvent.type(searchInput, filterValue);
        await waitFor(() => {
            expect(dispatch).toHaveBeenLastCalledWith(
                loadFilteredAndPagedHistoryLogs({
                    page: 0,
                    logsFilter: {
                        containerId,
                        taskId,
                        filterSettings: { ...initialFilterSettings, filter: filterValue },
                    },
                }),
            );
        });
    });

    it('should clear filter when clear icon button has been clicked', async () => {
        renderComponent(defaultProps, {
            ...defaultState,
            drawerHistory: {
                ...defaultState.drawerHistory,
                logsFilter: {
                    ...defaultState.drawerHistory.logsFilter,
                    filterSettings: { ...defaultState.drawerHistory.logsFilter.filterSettings, filter: 'test' },
                },
            },
        });

        const clearButton = screen.getByLabelText('close icon button');
        userEvent.click(clearButton);

        await waitFor(() => {
            expect(dispatch).toHaveBeenLastCalledWith(
                loadFilteredAndPagedHistoryLogs({
                    page: 0,
                    logsFilter: { containerId, taskId, filterSettings: { ...initialFilterSettings, filter: '' } },
                }),
            );
        });
    });

    it('should toggle sort order when clicking the sort button', async () => {
        renderComponent();
        const sortBtn = getNewestFirstBtn();
        expect(sortBtn).toBeVisible();

        userEvent.click(sortBtn as HTMLElement);

        await waitFor(() => {
            expect(dispatch).toHaveBeenLastCalledWith(
                loadFilteredAndPagedHistoryLogs({
                    page: 0,
                    logsFilter: { containerId, taskId, filterSettings: { ...initialFilterSettings, dateAsc: true } },
                }),
            );
        });
    });

    it("should render 'Oldest first' button when dateAsc is set to true", () => {
        renderComponent(defaultProps, {
            ...defaultState,
            drawerHistory: {
                ...defaultState.drawerHistory,
                logsFilter: {
                    ...defaultState.drawerHistory.logsFilter,
                    filterSettings: { ...defaultState.drawerHistory.logsFilter.filterSettings, dateAsc: true },
                },
            },
        });
        expect(getOldestFirstBtn()).toBeVisible();
        expect(getNewestFirstBtn()).not.toBeInTheDocument();
    });

    it('should render data grouped by date', () => {
        renderComponent(defaultProps, {
            ...defaultState,
            drawerHistory: {
                ...defaultState.drawerHistory,
                logs: mockGroupActivityLogs,
            },
        });
        expect(screen.getByText(formatDate(mockGroupActivityLogs[0].eventTime).toString())).toBeVisible();
        expect(screen.getByText(formatDate(mockGroupActivityLogs[1].eventTime).toString())).toBeVisible();
        expect(screen.getByText(getActivityTypeLabel(mockGroupActivityLogs[0].activityType))).toBeVisible();
        expect(screen.getByText(getActivityTypeLabel(mockGroupActivityLogs[1].activityType))).toBeVisible();
        expect(screen.getByText(getActivityTypeLabel(mockGroupActivityLogs[2].activityType))).toBeVisible();
    });

    it('should render correct data when loading and no data', () => {
        renderComponent(defaultProps, {
            ...defaultState,
            drawerHistory: {
                ...defaultState.drawerHistory,
                logs: [],
                isLoading: true,
            },
        });

        const loadingRows = screen.getAllByTestId('avatar-skeleton');
        expect(loadingRows).toHaveLength(DRAWER_HISTORY_NUMBER_OF_SKELETONS);
    });

    it('should load more logs when observer triggers intersect', () => {
        mockIntersectionObserverWithIntersectingFlag(true);
        renderComponent();

        expect(dispatch).toHaveBeenLastCalledWith(
            loadFilteredAndPagedHistoryLogs({
                page: 1,
                logsFilter: { containerId, taskId, filterSettings: initialFilterSettings },
            }),
        );
    });

    it('should render empty state when there are no logs and filters', () => {
        renderComponent(defaultProps, {
            ...defaultState,
            drawerHistory: {
                ...defaultState.drawerHistory,
                logs: [],
                isLoading: false,
            },
        });
        expect(screen.getByText('No history')).toBeVisible();
        expect(screen.getByText('Changes made to the task will appear here once available.')).toBeVisible();
    });

    it('should render empty filter state when there are no logs and but filters are present', () => {
        renderComponent(defaultProps, {
            ...defaultState,
            drawerHistory: {
                ...defaultState.drawerHistory,
                logs: [],
                isLoading: false,
                logsFilter: {
                    ...defaultState.drawerHistory.logsFilter,
                    filterSettings: { ...defaultState.drawerHistory.logsFilter.filterSettings, filter: 'xyz123' },
                },
            },
        });
        expect(screen.getByText('No results found')).toBeVisible();
        expect(screen.getByText('No results match the current filter selection. Please, adjust the filters and attempt your search again.')).toBeVisible();
    });
});
