import React from 'react';
import { DotLink, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import IdsFactory from '@xlr-ui/app/js/util/ids';
import { getReleaseHref, getReleaseVariableHref, getTargetTypeLabel, getTaskWithOpenedModalHref, removeAllAfterLastTask } from '../../helper';
import { ActivityLogsContainerEnum } from '../../types';

export interface ActivityLogsTypeReleaseProps {
    targetId: string | null;
    targetType: string | null;
    validTargetIds: string[];
}

const idsFactory = IdsFactory();

export const ActivityLogsTypeRelease = ({ targetId, targetType, validTargetIds }: ActivityLogsTypeReleaseProps) => {
    if (!targetId || !targetType) {
        return (
            <DotTypography data-testid="type-label-typography" variant="body1">
                <DotTooltip
                    placement="left"
                    title="Link is unavailable because the target item cannot be identified or referenced for releases and templates older than version 9.0.0."
                >
                    <>Not available</>
                </DotTooltip>
            </DotTypography>
        );
    }

    const renderNotFound = (label: string) => {
        return (
            <DotTypography data-testid="type-label-typography" variant="body1">
                <DotTooltip placement="left" title="Link is unavailable because the target item has been deleted or moved.">
                    <>{label}</>
                </DotTooltip>
            </DotTypography>
        );
    };

    const internalId = idsFactory.toInternalId(targetId);

    if (targetType === 'xlrelease.Release') {
        const releaseId = idsFactory.releaseIdFrom(internalId);
        return <DotLink href={getReleaseHref(releaseId)}>Release</DotLink>;
    } else if (targetId && idsFactory.isVariableId(internalId)) {
        const releaseId = idsFactory.releaseIdFrom(internalId);
        if (validTargetIds.includes(targetId)) {
            return <DotLink href={getReleaseVariableHref(releaseId, idsFactory.getName(targetId))}>Variable</DotLink>;
        } else {
            return renderNotFound('Variable');
        }
    } else {
        const label = getTargetTypeLabel(targetType, ActivityLogsContainerEnum.RELEASE);
        const cleanTaskId = removeAllAfterLastTask(internalId);
        if (idsFactory.isTaskId(cleanTaskId)) {
            const task = validTargetIds.includes(cleanTaskId);
            if (!task) {
                return renderNotFound(label);
            } else {
                const releaseId = idsFactory.releaseIdFrom(internalId);
                const phaseTaskId = idsFactory.phaseTaskIdFrom(internalId);
                const cleanPhaseTaskId = removeAllAfterLastTask(phaseTaskId);
                return <DotLink href={getTaskWithOpenedModalHref(releaseId, cleanPhaseTaskId)}>{label}</DotLink>;
            }
        } else if (label === 'Task') {
            return (
                <DotTypography data-testid="type-label-typography" variant="body1">
                    <DotTooltip placement="left" title={`Link is unavailable because the target item cannot be identified or referenced (${targetType}).`}>
                        <>{label}</>
                    </DotTooltip>
                </DotTypography>
            );
        } else {
            return (
                <DotTypography data-testid="type-label-typography" variant="body1">
                    <>{label}</>
                </DotTypography>
            );
        }
    }
};
