import React, { Component } from 'react';
import PropTypes from 'prop-types';
import clone from 'lodash/clone';
import { XlReactWidgetDate, XlReactWidgetLabel } from 'xl-react-components';
import moment from 'moment';
import './widget-datetime.less';

export class WidgetDateTime extends Component {
    static propTypes = {
        disabled: PropTypes.bool,
        metadata: PropTypes.object,
        minDate: PropTypes.any,
        model: PropTypes.any,
        onDatePristinityChange: PropTypes.func.isRequired,
        onModelChange: PropTypes.func.isRequired,
        onTimePristinityChange: PropTypes.func.isRequired,
        onValidityChange: PropTypes.func.isRequired,
    };

    constructor(props) {
        super(props);
        this.state = {
            datePristine: true,
            timePristine: true,
        };

        this.isDateValid = false;
        this.isTimeValid = true;
    }

    onStartDateChange = (selectedDate) => {
        if (_.isDate(selectedDate) && !_.isNaN(selectedDate.getDate())) {
            const { model, onModelChange } = this.props;
            const startDate = clone(model);
            startDate.setFullYear(selectedDate.getFullYear(), selectedDate.getMonth(), selectedDate.getDate());

            if (!moment(startDate).isSame(moment(model))) {
                onModelChange(startDate);
            }
        }
    };

    onStartTimeChange = (selectedTime) => {
        if (!selectedTime) {
            this.onTimeValidityChange(false);
            return;
        }
        this.onTimeValidityChange(true);
        const time = selectedTime.split(':');
        const { model, onModelChange } = this.props;
        const startDate = clone(model);
        startDate.setHours(time[0], time[1], 0, 0);

        if (!moment(startDate).isSame(moment(model))) {
            this.onTimePristine(false);
            onModelChange(startDate);
        }
    };

    onDatePristine = (pristine) => {
        const { onDatePristinityChange } = this.props;

        this.setState({ datePristine: pristine });
        onDatePristinityChange(pristine);
    };

    onTimePristine = (pristine) => {
        const { onTimePristinityChange } = this.props;

        this.setState({ timePristine: pristine });
        onTimePristinityChange(pristine);
    };

    get isFormValid() {
        return this.isDateValid && this.isTimeValid;
    }

    onTimeValidityChange = (valid) => {
        const { onValidityChange } = this.props;

        this.isTimeValid = valid;
        onValidityChange(this.isFormValid);
    };

    onDateValidityChange = (valid) => {
        const { onValidityChange } = this.props;

        this.isDateValid = valid;
        onValidityChange(this.isFormValid);
    };

    render() {
        const { disabled, metadata, minDate, model } = this.props;
        const currentTime = model && model.toLocaleTimeString('en-GB', { hour: '2-digit', minute: '2-digit', hour12: false });
        return (
            <div className="xl-react-components widget-datetime">
                {metadata.label && <XlReactWidgetLabel metadata={metadata} />}
                <div className="date-time-wrapper">
                    <div className="date-time-input">
                        <XlReactWidgetDate
                            dateFormat={metadata.dateFormat}
                            disabled={disabled}
                            minDate={minDate}
                            model={model}
                            onModelChange={this.onStartDateChange}
                            onPristinityChange={this.onDatePristine}
                            onValidityChange={this.onDateValidityChange}
                        />
                        <div className="xl-react-components xl-react-widget-time">
                            <div className="xl-react-components-input-full">
                                <input onChange={(e) => this.onStartTimeChange(e.target.value)} readOnly={disabled} required type="time" value={currentTime} />
                            </div>
                        </div>
                    </div>
                    <div className="xl-react-components-description">
                        <div>{metadata.description}</div>
                    </div>
                </div>
            </div>
        );
    }
}
