import { all, call, put, select, takeLatest } from 'redux-saga/effects';
import ToastrFactory from '@xlr-ui/app/js/util/toastrFactory';
import isEmpty from 'lodash/isEmpty';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import { Environment, EnvironmentEdit, EnvironmentName } from '../../../application-list.types';
import { httpDELETE, httpPOST } from '@xlr-ui/app/features/common/services/http';
import { environmentLists } from './environment-list.reducer';
import { PayloadAction } from '@reduxjs/toolkit';
import { IStateService } from 'angular-ui-router';
import { getEnvironments } from './environment-list.selectors';

export const toastr = ToastrFactory();

const { createEnvironment, deleteEnvironment, editEnvironment, initEnvironmentList, searchEnvironments, setEnvironments, setLoading, setStages } =
    environmentLists.actions;
export function* onSearch(action: { payload: EnvironmentName }) {
    const { title, stageTitle, folderId } = action.payload;
    const stage = stageTitle || null;

    try {
        const { data: environments } = yield call(httpPOST, 'api/v1/environments/search', { title, stage, folderId });
        yield put(setEnvironments(environments));
    } catch (e) {
        // eslint-disable-next-line
        yield call(console.error, 'Environment-list-saga-search-envs: ', e);
    }
}

export function* onAddEnvironment(action: PayloadAction<string>) {
    const folderId = action.payload;
    const $state: IStateService = yield call(getAngularService, '$state');
    if (isEmpty(folderId)) {
        $state.go('environmentCreate');
    } else {
        $state.go('folders.detail.environmentCreate');
    }
}

export function* onEditEnvironment(action: PayloadAction<EnvironmentEdit>) {
    const { envId, folderId } = action.payload;
    const $state: IStateService = yield call(getAngularService, '$state');
    if (isEmpty(folderId)) {
        $state.go('environmentCreate', { environmentId: envId });
    } else {
        $state.go('folders.detail.environmentCreate', { environmentId: envId });
    }
}

export function* onDeleteEnvironment(action: PayloadAction<Environment>) {
    const environment = action.payload;
    try {
        yield put(setLoading(true));
        yield call(httpDELETE, `api/v1/environments/${environment.id}`);
        yield put(setLoading(false));

        const envs: Array<Environment> = yield select(getEnvironments);
        yield put(setEnvironments(envs.filter((env) => env.id !== environment.id)));
        yield call(toastr.success, `Environment [${environment.title}] deleted.`);
    } catch (e) {
        // eslint-disable-next-line
        yield call(console.error, 'Environment-list-saga-delete-env: ', e);
    }
}

export function* onInitEnvironmentList(action: PayloadAction<string>) {
    try {
        const folderId = action.payload;
        const { data: environments } = yield call(httpPOST, 'api/v1/environments/search', { title: '', stage: null, folderId });
        const { data: stages } = yield call(httpPOST, 'api/v1/environments/stages/search', { title: '' });
        yield put(setEnvironments(environments));
        yield put(setStages(stages));
    } catch (e) {
        // eslint-disable-next-line
        yield call(console.error, 'Environment-list-saga-initialization: ', e);
    }
}

export default function* environmentListSaga() {
    yield all([
        takeLatest(createEnvironment, onAddEnvironment),
        takeLatest(editEnvironment, onEditEnvironment),
        takeLatest(deleteEnvironment, onDeleteEnvironment),
        takeLatest(searchEnvironments, onSearch),
        takeLatest(initEnvironmentList, onInitEnvironmentList),
    ]);
}
