import React, { Fragment, useEffect } from 'react';
import toPairs from 'lodash/toPairs';
import { CssGrid, DotEmptyFilterState, DotEmptyState, DotIconButton, DotSearch, DotTypography } from '@digital-ai/dot-components';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';
import { drawerHistory, getDrawerHistoryState } from '../ducks/drawer-history.reducer';
import { ActivityLog, ActivityLogsFilterSettings } from '../../../types';
import { GridCellFullWidth } from '@xlr-ui/app/features/tasks/components/rails/grid.layout';
import IdsFactory from '@xlr-ui/app/js/util/ids';
import { TaskDrawerContentCustomProps } from '@xlr-ui/app/features/tasks/components/content/task-drawer-content.component';
import { DrawerHistoryRow } from './drawer-history-row.component';
import { groupActivityLogsByDate } from '../../../helper';
import { DividerWithDatePill } from '@xlr-ui/app/features/common/components/divider-with-date-pill/divider-with-date-pill.component';
import { DRAWER_HISTORY_NUMBER_OF_LOADING_ROWS, DRAWER_HISTORY_NUMBER_OF_SKELETONS } from '../constants';
import { useInfiniteScrollPaginationWithContainer } from '@xlr-ui/app/features/common/hooks/infinite-scroll-pagination-with-container.hook';
import { DrawerHistorySkeletonRow } from './drawer-history-skeleton-row.component';
import './drawer-history.component.less';

const { init, loadFilteredAndPagedHistoryLogs, reset } = drawerHistory.actions;

export type DrawerHistoryProps = TaskDrawerContentCustomProps;

const Ids = IdsFactory();

const SCROLL_CONTAINER_CLASS_NAME = 'drawer-history-content';

export const DrawerHistory = ({ release, task }: DrawerHistoryProps) => {
    const dispatch = useAppDispatch();
    const { logsFilter, logs, page, isLoading, isLastPage } = useAppSelector(getDrawerHistoryState);
    const taskId = task.id;
    const filterSettings = { ...logsFilter.filterSettings, targetId: Ids.toDomainId(taskId) };
    const containerId = release.id;
    const isDateAsc = filterSettings.dateAsc;
    const sortIcon = isDateAsc ? 'sort-asc' : 'sort-desc';
    const sortTooltip = isDateAsc ? 'Oldest first' : 'Newest first';
    const logsGroupedByEventTime = toPairs(groupActivityLogsByDate(logs));
    const isFirstPage = page === 0;
    const hasFilters = !!filterSettings.filter;
    const defaultLogsFilter = {
        ...logsFilter,
        containerId,
        taskId,
    };

    useEffect(() => {
        dispatch(init({ containerId, taskId, filterSettings }));
        return () => dispatch(reset());
    }, []);

    const handleIntersect = () => {
        if (isLoading || isLastPage) return;
        dispatch(loadFilteredAndPagedHistoryLogs({ page: page + 1, logsFilter: defaultLogsFilter }));
    };

    const handleFilterChange = (currentFilterSettings: ActivityLogsFilterSettings) => {
        dispatch(
            loadFilteredAndPagedHistoryLogs({
                page: 0,
                logsFilter: { ...defaultLogsFilter, filterSettings: currentFilterSettings },
            }),
        );
    };

    const { observerTarget } = useInfiniteScrollPaginationWithContainer<ActivityLog, HTMLDivElement>({
        isFirstPage,
        items: logs,
        onIntersect: handleIntersect,
        scrollContainerSelector: `.${SCROLL_CONTAINER_CLASS_NAME}`,
    });

    const renderHistoryRows = () => {
        if (isLoading && !logsGroupedByEventTime.length) {
            return [...Array(DRAWER_HISTORY_NUMBER_OF_SKELETONS)].map((_, index: number) => <DrawerHistorySkeletonRow key={index} />);
        } else if (logsGroupedByEventTime.length) {
            const rows = logsGroupedByEventTime.map(([eventTime, currentActivityLogs], index) => {
                return (
                    <Fragment key={`${eventTime}-${index}`}>
                        <DividerWithDatePill date={parseInt(eventTime)} />
                        {currentActivityLogs.map((log, logIndex) => {
                            return <DrawerHistoryRow key={`${log.eventTime}-${logIndex}`} log={log} />;
                        })}
                    </Fragment>
                );
            });
            !isLastPage &&
                rows.push(
                    ...Array(DRAWER_HISTORY_NUMBER_OF_LOADING_ROWS)
                        .fill(undefined)
                        .map((_, index: number) => {
                            const isObserverRow = index === 0;
                            return <DrawerHistorySkeletonRow key={index} observerTarget={isObserverRow ? observerTarget : undefined} />;
                        }),
                );
            return rows;
        } else {
            return hasFilters ? (
                <DotEmptyFilterState />
            ) : (
                <DotEmptyState illustrationId="empty" subtitle="Changes made to the task will appear here once available." title="No history" />
            );
        }
    };

    return (
        <CssGrid className="drawer-history-grid">
            <GridCellFullWidth className="drawer-history-grid-cell" expanded={false}>
                <div className="drawer-history-content-wrapper">
                    <div className="drawer-history-content-header">
                        <DotTypography variant="h3">History</DotTypography>
                        <div className="drawer-history-content-header-controls">
                            <DotSearch
                                id="drawer-history-filter"
                                onClear={() => handleFilterChange({ ...filterSettings, filter: '' })}
                                onSearch={(searchText) => handleFilterChange({ ...filterSettings, filter: searchText })}
                                placeholder="Filter by user or activity"
                                value={filterSettings.filter}
                            />
                            <DotIconButton
                                data-testid="drawer-history-sort-btn"
                                iconId={sortIcon}
                                onClick={() => handleFilterChange({ ...filterSettings, dateAsc: !isDateAsc })}
                                tooltip={sortTooltip}
                            />
                        </div>
                    </div>
                    <div className={SCROLL_CONTAINER_CLASS_NAME}>{renderHistoryRows()}</div>
                </div>
            </GridCellFullWidth>
        </CssGrid>
    );
};
