import { all, call, put, select, takeLatest } from 'redux-saga/effects';
import { SagaIterator } from 'redux-saga';
import drawerHistorySaga, { addHistoryRailSaga, initSaga, loadFilteredAndPagedHistoryLogsSaga, loadLogs } from './drawer-history.saga';
import { drawerHistory, getDrawerHistoryState } from './drawer-history.reducer';
import { bootstrap } from '@xlr-ui/app/features/bootstrap/ducks/bootstrap-reducer';
import { ActivityLog } from '../../../types';
import { DEFAULT_FILTER_SETTINGS } from '../../../constants';
import { getFilterSettingsForLoadPayload } from '../../../helper';
import { httpPOST } from '@xlr-ui/app/features/common/services/http';
import { DRAWER_HISTORY_ITEMS_PER_PAGE } from '../constants';
import { taskDrawer } from '@xlr-ui/app/features/tasks/ducks/task-drawer.reducer';

const { init, loadFilteredAndPagedHistoryLogs, setPage, setIsLoading, setLogs, setLogsFilter, setIsLastPage } = drawerHistory.actions;
const { addRail } = taskDrawer.actions;
const { afterLogin } = bootstrap.actions;

describe('drawerHistory saga', () => {
    const logs: ActivityLog[] = [
        {
            user: {
                username: 'admin',
                fullName: 'Release Administrator',
                email: 'releaseadmin@dai.com',
            },
            activityType: 'RELEASE_STARTED',
            message: 'Release started',
            eventTime: 1726472112658,
            targetType: 'time.Schedule',
            targetId: 'Applications/FolderSamplesAndTutorials/Triggerf5d982a3d9724f8888e28167cd68ebef',
            dataId: null,
        },
    ];

    const logsFilterMock = {
        containerId: '123',
        filterSettings: DEFAULT_FILTER_SETTINGS,
        taskId: 'Task1',
    };

    it('should yield all effects', () => {
        const gen: SagaIterator = drawerHistorySaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(init, initSaga),
                takeLatest(afterLogin, addHistoryRailSaga),
                takeLatest(loadFilteredAndPagedHistoryLogs, loadFilteredAndPagedHistoryLogsSaga),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('initSaga', () => {
        it('should call proper generators on init saga for release', () => {
            const generator: SagaIterator = initSaga({ payload: logsFilterMock, type: 'INIT' });
            expect(generator.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(generator.next().value).toStrictEqual(put(setPage(0)));
            expect(generator.next().value).toStrictEqual(call(loadLogs, 0, logsFilterMock));
            expect(generator.next(logs).value).toStrictEqual(put(setIsLastPage(true)));
            expect(generator.next().value).toStrictEqual(put(setLogs(logs)));
            expect(generator.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(generator.next().done).toBe(true);
        });
    });

    describe('loadFilteredAndPagedHistoryLogsSaga', () => {
        const oldLogs: ActivityLog[] = [
            {
                user: {
                    username: 'admin',
                    fullName: 'Test',
                    email: 'releaseadmin@dai.com',
                },
                activityType: 'RELEASE_STARTED',
                message: 'Release started',
                eventTime: 1726472112644,
                targetType: 'time.Schedule',
                targetId: 'Applications/FolderSamplesAndTutorials/Triggerf5d982a3d9724f8888e28167cd68ebef',
                dataId: null,
            },
        ];

        const verifyFilterFlow = (page: number, currentLogs: ActivityLog[]) => {
            const logsFilter = { filter: 'test' } as never;
            const generator: SagaIterator = loadFilteredAndPagedHistoryLogsSaga({
                payload: { logsFilter, page },
                type: 'LOAD_FILTERED_AND_PAGED_HISTORY_LOGS_SAGA',
            });
            expect(generator.next().value).toStrictEqual(select(getDrawerHistoryState));
            expect(generator.next({ logs: currentLogs }).value).toStrictEqual(put(setIsLoading(true)));
            if (page === 0 && currentLogs.length > 0) {
                expect(generator.next().value).toStrictEqual(put(setLogs([])));
            }
            expect(generator.next().value).toStrictEqual(put(setPage(page)));
            expect(generator.next().value).toStrictEqual(put(setLogsFilter(logsFilter)));
            expect(generator.next().value).toStrictEqual(call(loadLogs, page, logsFilter));
            expect(generator.next(logs).value).toStrictEqual(put(setIsLastPage(true)));
            if (page > 0) {
                expect(generator.next().value).toStrictEqual(put(setLogs(currentLogs.concat(logs))));
            } else {
                expect(generator.next().value).toStrictEqual(put(setLogs(logs)));
            }
            expect(generator.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(generator.next().done).toBe(true);
        };

        it('should call proper generators when filters provided with page 0 and logs present', () => {
            verifyFilterFlow(0, oldLogs);
        });

        it('should proper generators when not on the first page', () => {
            verifyFilterFlow(1, oldLogs);
        });
    });

    describe('loadLogs', () => {
        it('should call proper generators', () => {
            const page = 2;
            const generator: SagaIterator = loadLogs(page, logsFilterMock);
            expect(generator.next().value).toEqual(
                call(
                    httpPOST,
                    `activity/${logsFilterMock.containerId}/logs/search?page=${page}&resultsPerPage=${DRAWER_HISTORY_ITEMS_PER_PAGE}`,
                    getFilterSettingsForLoadPayload(logsFilterMock.filterSettings),
                ),
            );
            expect(generator.next(logs).done).toBe(true);
        });
    });

    describe('addHistoryRailSaga', () => {
        it('should call proper generators', () => {
            const generator: SagaIterator = addHistoryRailSaga();
            expect(generator.next().value).toStrictEqual(put(addRail({ title: 'History', iconId: 'history', index: 6 })));
            expect(generator.next(logs).done).toBe(true);
        });
    });
});
