import React, { ReactElement } from 'react';
import { ReactWrapper } from 'enzyme';
import { stringPropertyMock } from '@xlr-ui/app/features/tasks/__mocks__/property.mock';
import { taskMock } from '@xlr-ui/app/features/tasks/__mocks__/task.mock';
import { mountWithStoreAndTheme } from '@xlr-ui/tests/unit/testing-utils';
import { PropertiesEnum, TaskInputCommonProps } from '@xlr-ui/app/features/tasks/components/rails/config/components/types';
import { StringWithVariables } from './string-with-variables.component';
import { TaskVariableAutocomplete } from '@xlr-ui/app/features/tasks/components/rails/config/components/variable-autocomplete/task-variable-autocomplete.component';
import { VariableToggler } from '@xlr-ui/app/features/tasks/components/rails/config/components/variable-toggler/variable-toggler.component';
import * as angularAccessor from '@xlr-ui/app/features/common/services/angular-accessor';
import { MetadataService } from '@xlr-ui/app/features/tasks/types/angular';

describe('StringWithVariables', () => {
    const state = {
        taskDrawer: {
            allVariables: [{ key: 'var' }],
        },
    };
    const task = {
        ...taskMock,
        inputProperties: {
            string: '${myVar}',
        },
    };
    const onChange = jest.fn();
    const dispatch = jest.fn();
    let wrapper: ReactWrapper;

    const defaultProps: TaskInputCommonProps = {
        task,
        propertiesEnum: PropertiesEnum.InputProperties,
        property: stringPropertyMock,
        disabled: false,
        expanded: false,
        onChange,
    };

    const getVariableToggler = () => wrapper.find(VariableToggler);
    const getVariableAutocomplete = () => wrapper.find(TaskVariableAutocomplete);

    const mount = (component: ReactElement) => {
        wrapper = mountWithStoreAndTheme(component, dispatch, state);
    };

    const mountStringWithVariables = (props: TaskInputCommonProps = defaultProps) => {
        mount(
            <StringWithVariables {...props}>
                <h1>hi</h1>
            </StringWithVariables>,
        );
    };

    beforeEach(() => {
        const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance<MetadataService, [name: unknown]>;
        getAngularServiceSpy.mockReturnValue({
            getSubDescriptors: (_a: string) => Promise.resolve([]),
        } as never);
    });

    it('should create variable toggle', () => {
        mountStringWithVariables();
        const toggleButton = getVariableToggler();
        const toggleButtonProps = toggleButton.props();
        expect(toggleButton).toExist();
        expect(toggleButtonProps.buttonOptions).toMatchObject([
            {
                ariaLabel: 'Click to switch between variable and manual entry',
                disabled: false,
                iconId: 'variable',
                text: '',
                value: true,
            },
        ]);
        expect(toggleButtonProps.isInVariableSelection).toBe(true);
    });
    it('should not create variable toggle if is disabled', () => {
        const props = {
            ...defaultProps,
            disabled: true,
        };
        mountStringWithVariables(props);
        const toggleButton = getVariableToggler();
        expect(toggleButton).not.toExist();
    });

    it('should toggle and clean the value', () => {
        mountStringWithVariables();
        expect(getVariableAutocomplete()).toExist();
        const variableToggler = getVariableToggler();
        variableToggler.invoke('onSwitchToVariableButtonClick')();
        expect(onChange).toHaveBeenCalledWith(null);
        expect(getVariableAutocomplete()).not.toExist();
    });

    it('should select variable toggle if value is variable', () => {
        mountStringWithVariables();
        expect(getVariableAutocomplete()).toExist();
        expect(getVariableToggler().props().isInVariableSelection).toBe(true);
    });

    it('should pass props to VariableAutocomplete', () => {
        mountStringWithVariables();
        const variableAutocompleteProps = getVariableAutocomplete().props();
        expect(variableAutocompleteProps.disabled).toBe(false);
        expect(variableAutocompleteProps.property).toBe(stringPropertyMock);
        expect(variableAutocompleteProps.variables).toMatchObject([{ key: 'var' }]);
        expect(variableAutocompleteProps.taskProperties).toMatchObject({
            string: {
                value: null,
                variable: '${myVar}',
            },
        });
    });

    it('should update variable selected', () => {
        mountStringWithVariables();
        const variableAutocompleteProps = getVariableAutocomplete().props();
        variableAutocompleteProps.onChange({
            value: null,
            variable: '${other}',
        });
        expect(onChange).toHaveBeenCalledWith('${other}');
    });

    it('should create new variable', () => {
        mountStringWithVariables();
        const variableAutocompleteProps = getVariableAutocomplete().props();
        variableAutocompleteProps.onAddVariable('variable', 'xlrelease.StringVariable');
        expect(dispatch).toHaveBeenCalledWith({
            payload: {
                propertiesEnum: PropertiesEnum.InputProperties,
                propertyName: stringPropertyMock.name,
                taskValue: '${variable}',
                variableName: 'variable',
                variableType: 'xlrelease.StringVariable',
            },
            type: 'propertiesForm/createReleaseVariableAndUpdateTask',
        });
    });
});
