import React from 'react';
import { User, WithTitleAndId } from '@xlr-ui/app/types';
import { getSelectedCategoryFromActivitySettings, getSelectedCategoryFromIds } from '../helper';
import { FilterGroup } from '@xlr-ui/app/features/common/components/filter/filter-group.component';
import { FilterAutocomplete } from './filters/filter-autocomplete.component';
import { FilterDate } from '@xlr-ui/app/features/common/components/filter/filter-date.component';
import { FilterUsername } from '@xlr-ui/app/features/common/components/filter/filter-username.component';
import { ActivityLogsFilterSettings } from '../../../types';
import { FilterDrawer } from '@xlr-ui/app/features/common/components/filter-drawer/filter-drawer.component';

export interface ActivityLogsFilterDrawerProps {
    activityTypes: WithTitleAndId[];
    categories: WithTitleAndId[];
    filter: ActivityLogsFilterSettings;
    isDrawerOpened: boolean;
    numberOfFiltersApplied: number;
    onClearAll: () => void;
    onDrawerClose: () => void;
    onFilterChange: (filter: ActivityLogsFilterSettings) => void;
    users: User[];
}

export const ActivityLogsFilterDrawer = ({
    isDrawerOpened,
    numberOfFiltersApplied,
    onClearAll,
    onDrawerClose,
    onFilterChange,
    filter,
    categories,
    activityTypes,
    users,
}: ActivityLogsFilterDrawerProps) => {
    return (
        <FilterDrawer isOpened={isDrawerOpened} numberOfFiltersApplied={numberOfFiltersApplied} onClearAllFilters={onClearAll} onClose={onDrawerClose}>
            <>
                <FilterGroup title="Activity">
                    <FilterAutocomplete
                        inputId="activityTypes"
                        onChange={(newActivityTypes) =>
                            onFilterChange({
                                ...filter,
                                activityTypes: newActivityTypes,
                            })
                        }
                        options={activityTypes}
                        placeholder="Filter by activity type"
                        value={activityTypes.filter((at) => filter.activityTypes.includes(at.id))}
                    />
                </FilterGroup>
                <FilterGroup title="Performed by">
                    <FilterUsername
                        inputId="usernames"
                        onChange={(usernames) => onFilterChange({ ...filter, usernames: usernames as string[] })}
                        placeholder="Filter by user or system"
                        users={users}
                        value={filter.usernames}
                    />
                </FilterGroup>
                <FilterGroup title="Category">
                    <FilterAutocomplete
                        className="categories-autocomplete"
                        inputId="categories"
                        onChange={(newCategories) => onFilterChange({ ...filter, ...getSelectedCategoryFromIds(newCategories) })}
                        options={categories}
                        placeholder="Filter by category"
                        value={categories.filter((c) => getSelectedCategoryFromActivitySettings(filter).includes(c.id))}
                    />
                </FilterGroup>
                <FilterGroup title="Time from">
                    <FilterDate
                        inputId="time-from"
                        onChange={(from) =>
                            onFilterChange({
                                ...filter,
                                from,
                            })
                        }
                        value={filter.from}
                    />
                </FilterGroup>
                <FilterGroup title="Time to">
                    <FilterDate inputId="time-to" onChange={(to) => onFilterChange({ ...filter, to })} value={filter.to} />
                </FilterGroup>
            </>
        </FilterDrawer>
    );
};
