import React from 'react';
import { DotLink, DotTypography, DotTooltip } from '@digital-ai/dot-components';
import { mountWithTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { ActivityLogsTypeRelease, ActivityLogsTypeReleaseProps } from './activity-logs-type.release.component';

describe('ActivityLogsTypeRelease component', () => {
    let wrapper: ReactWrapper;

    const validTargetIds = ['Folder1-Release1-Phase1-Task1', 'Applications/Folder1/Release1/Variable1'];

    const getTypeLabelTypography = () => wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'type-label-typography');
    const getDotLink = () => wrapper.find(DotLink);
    const getTooltip = () => wrapper.find(DotTooltip);
    const mount = (props: ActivityLogsTypeReleaseProps) => {
        wrapper = mountWithTheme(<ActivityLogsTypeRelease {...props} />);
    };

    afterAll(() => {
        jest.restoreAllMocks();
    });

    beforeEach(() => {
        window.location.hash = '#/release/Folder1-Release1/logs';
    });

    it('should render links for release events', () => {
        mount({ targetId: 'Applications/Folder1/Release1', targetType: 'xlrelease.Release', validTargetIds });
        expect(getTypeLabelTypography()).not.toExist();
        const dotLink = getDotLink();
        expect(dotLink).toExist();
        expect(dotLink.text()).toBe('Release');
        expect(dotLink.props().href).toBe('#/release/Folder1-Release1');
    });

    it('should render typography for phase events', () => {
        mount({ targetId: 'Applications/Folder1/Release1/Phase1', targetType: 'xlrelease.Phase', validTargetIds });
        expect(getDotLink()).not.toExist();
        const typeLabelTypography = getTypeLabelTypography();
        expect(typeLabelTypography.text()).toBe('Phase');
        expect(typeLabelTypography.props().variant).toBe('body1');
    });

    it('should render link for task comments', () => {
        mount({ targetId: 'Applications/Folder1/Release1/Phase1/Task1/Comment', targetType: 'xlrelease.Task', validTargetIds });
        expect(getTypeLabelTypography()).not.toExist();
        const dotLink = getDotLink();
        expect(dotLink).toExist();
        expect(dotLink.text()).toBe('Task');
        expect(dotLink.props().href).toBe('#/release/Folder1-Release1?openTaskDetailsModal=Phase1-Task1');
    });

    it('should render link for existing tasks', () => {
        mount({ targetId: 'Applications/Folder1/Release1/Phase1/Task1', targetType: 'xlrelease.Task', validTargetIds });
        const dotLink = getDotLink();
        expect(dotLink).toExist();
        expect(dotLink.text()).toBe('Task');
        expect(dotLink.props().href).toBe('#/release/Folder1-Release1?openTaskDetailsModal=Phase1-Task1');
        expect(getTypeLabelTypography()).not.toExist();
    });

    it('should render typography and tooltip for non existing tasks', () => {
        mount({ targetId: 'Applications/Folder1/Release1/Phase1/Task1', targetType: 'xlrelease.Task', validTargetIds: [] });
        const typeLabelTypography = getTypeLabelTypography();
        expect(typeLabelTypography).toExist();
        expect(typeLabelTypography.text()).toBe('Task');
        expect(typeLabelTypography.props().variant).toBe('body1');
        expect(getTooltip().props().title).toBe('Link is unavailable because the target item has been deleted or moved.');
    });

    it('should render link for existing variables', () => {
        mount({ targetId: 'Applications/Folder1/Release1/Variable1', targetType: 'xlrelease.StringVariable', validTargetIds });
        const dotLink = getDotLink();
        expect(dotLink).toExist();
        expect(dotLink.text()).toBe('Variable');
        expect(dotLink.props().href).toBe('#/release/Folder1-Release1/variables?variable=Variable1');
        expect(getTypeLabelTypography()).not.toExist();
    });

    it('should render typography and tooltip for non existing variables', () => {
        mount({ targetId: 'Applications/Folder1/Release1/Variable1', targetType: 'xlrelease.StringVariable', validTargetIds: [] });
        const typeLabelTypography = getTypeLabelTypography();
        expect(typeLabelTypography).toExist();
        expect(typeLabelTypography.text()).toBe('Variable');
        expect(typeLabelTypography.props().variant).toBe('body1');
        expect(getTooltip().props().title).toBe('Link is unavailable because the target item has been deleted or moved.');
    });

    it('should render correct link for triggers with complex paths', () => {
        const complexTriggerId = 'Applications/Folder123/Folder456/Release789/Trigger0123456789ABCDEF';
        mount({
            targetId: complexTriggerId,
            targetType: 'unknown.AnyTrigger',
            validTargetIds: [], // Empty because triggers not validated against this array
        });

        const dotLink = getDotLink();
        expect(dotLink).toExist();
        expect(dotLink.text()).toBe('Trigger');
        expect(dotLink.props().href).toBe('#/trigger-management/Trigger0123456789ABCDEF');
    });

    it('should prioritize ID-based trigger detection over targetType classification', () => {
        mount({
            targetId: 'Applications/Folder1/Trigger123',
            targetType: 'xlrelease.SomeNonTriggerType', // targetType suggests it's NOT a trigger
            validTargetIds,
        });

        const dotLink = getDotLink();
        expect(dotLink).toExist();
        expect(dotLink.text()).toBe('Trigger'); // But ID detection makes it a trigger
        expect(dotLink.props().href).toBe('#/trigger-management/Trigger123');
        expect(getTypeLabelTypography()).not.toExist();
    });

    it('should render link for plugin trigger types regardless of specific targetType', () => {
        // Test with plugin trigger types - the key is ID detection, not targetType
        const pluginTriggerCases = [
            { targetType: 'git.Poll', triggerId: 'TriggerGit' },
            { targetType: 'customPlugin.MyTrigger', triggerId: 'TriggerCustom' },
            { targetType: 'unknown.FutureTrigger', triggerId: 'TriggerFuture' },
        ];

        pluginTriggerCases.forEach(({ targetType, triggerId }) => {
            mount({ targetId: `Applications/Folder1/${triggerId}`, targetType, validTargetIds });
            const dotLink = getDotLink();
            expect(dotLink).toExist();
            expect(dotLink.text()).toBe('Trigger');
            expect(dotLink.props().href).toBe(`#/trigger-management/${triggerId}`);
            expect(getTypeLabelTypography()).not.toExist();
        });
    });

    it('should render trigger links without "deleted/moved" tooltip', () => {
        const triggerTargetId = 'Applications/MyFolder/Trigger456';
        const triggerType = 'unknown.AnyTrigger';

        mount({
            targetId: triggerTargetId,
            targetType: triggerType,
            validTargetIds: [], // Note: validTargetIds doesn't contain triggers - this is by design
        });

        // Should render a clickable DotLink, NOT a DotTypography with error tooltip
        const dotLink = getDotLink();
        const typeLabelTypography = getTypeLabelTypography();

        // Verify we get a working link (success case)
        expect(dotLink).toExist();
        expect(dotLink.text()).toBe('Trigger');
        expect(dotLink.props().href).toBe('#/trigger-management/Trigger456');

        // Verify we DON'T get the error state (DotTypography with error tooltip)
        expect(typeLabelTypography).not.toExist();

        // Key test: Check that no error tooltip with the specific "deleted/moved" message exists
        const errorTooltip = wrapper.findWhere((node) => node.is(DotTooltip) && node.props().title && node.props().title.includes('deleted or moved'));
        expect(errorTooltip).not.toExist();
    });

    it('should render typography and tooltip for unknown events', () => {
        // i think this will never happen, leaving it here just in case
        mount({ targetId: `Applications/Folder1/Release1/Attachment`, targetType: 'xlrelease.UnknownSomething', validTargetIds });
        const typeLabelTypography = getTypeLabelTypography();
        expect(typeLabelTypography).toExist();
        expect(typeLabelTypography.text()).toBe('Task');
        expect(typeLabelTypography.props().variant).toBe('body1');
        expect(getTooltip().props().title).toBe('Link is unavailable because the target item cannot be identified or referenced (xlrelease.UnknownSomething).');
    });

    it('should render typography and not tooltip for task attachments', () => {
        mount({ targetId: `Applications/Folder1/Release1/Attachment1`, targetType: 'xlrelease.Attachment', validTargetIds });
        const typeLabelTypography = getTypeLabelTypography();
        expect(typeLabelTypography).toExist();
        expect(typeLabelTypography.text()).toBe('Attachment');
        expect(typeLabelTypography.props().variant).toBe('body1');
        expect(getTooltip()).not.toExist();
    });

    it('should render typography and tooltip for facet events', () => {
        mount({ targetId: `Facetf7ce544bb2b54d82bb19066a3f9755d0`, targetType: 'udm.BuildRecord', validTargetIds });
        const typeLabelTypography = getTypeLabelTypography();
        expect(typeLabelTypography).toExist();
        expect(typeLabelTypography.text()).toBe('Task');
        expect(typeLabelTypography.props().variant).toBe('body1');
        expect(getTooltip().props().title).toBe('Link is unavailable because the target item cannot be identified or referenced (udm.BuildRecord).');
    });

    it('should render "not available" message if there is no targetId', () => {
        mount({ targetId: null, targetType: null, validTargetIds: [] });
        expect(getDotLink()).not.toExist();
        expect(getTypeLabelTypography()).toIncludeText('Not available');
        expect(getTooltip().props().title).toBe(
            'Link is unavailable because the target item cannot be identified or referenced for releases and templates older than version 9.0.0.',
        );
    });
});
