import isBoolean from 'lodash/isBoolean';
import { ActivityLogColumnProps, ActivityLogsContainerEnum } from '../types';
import { DELIVERY_ACTIVITY_LOG_CATEGORIES, IMPORTANT_ACTIVITY_TYPES, RELEASE_ACTIVITY_LOG_FILTERS, TRIGGER_ACTIVITY_LOG_CATEGORIES } from '../constants';
import { WithTitleAndId } from '@xlr-ui/app/types';
import { getPageTypeFromUrl } from '@xlr-ui/app/features/common/helper/path';
import { getCityFromClientTimezone } from '@xlr-ui/app/features/common/helper/date';
import { ActivityLogsFilterSettings } from '../../../types';
import { getActivityStyleAndLabel } from '../../../helper';

export const getActivityLogsCategories = (activityLogsContainerEnum: ActivityLogsContainerEnum): WithTitleAndId[] => {
    switch (activityLogsContainerEnum) {
        case ActivityLogsContainerEnum.DELIVERY:
            return DELIVERY_ACTIVITY_LOG_CATEGORIES;
        case ActivityLogsContainerEnum.RELEASE:
            return RELEASE_ACTIVITY_LOG_FILTERS;
        case ActivityLogsContainerEnum.TRIGGER:
            return TRIGGER_ACTIVITY_LOG_CATEGORIES;
        default:
            return RELEASE_ACTIVITY_LOG_FILTERS;
    }
};

export const getSelectedCategoryFromActivitySettings = (allFilterSettings: ActivityLogsFilterSettings): string[] => {
    return Object.keys(allFilterSettings).filter((key) => {
        const typedKey = key as keyof ActivityLogsFilterSettings;
        return isBoolean(allFilterSettings[typedKey]) && allFilterSettings[typedKey] === true;
    });
};

export const getSelectedCategoryFromIds = (
    ids: string[],
): Pick<
    ActivityLogsFilterSettings,
    | 'comments'
    | 'deliveryEdit'
    | 'execution'
    | 'lifecycle'
    | 'other'
    | 'reassign'
    | 'releaseEdit'
    | 'reportingRecordEdit'
    | 'security'
    | 'taskEdit'
    | 'triggerEdit'
> => {
    return {
        comments: ids.includes('comments'),
        deliveryEdit: ids.includes('deliveryEdit'),
        execution: ids.includes('execution'),
        lifecycle: ids.includes('lifecycle'),
        other: ids.includes('other'),
        reassign: ids.includes('reassign'),
        releaseEdit: ids.includes('releaseEdit'),
        reportingRecordEdit: ids.includes('reportingRecordEdit'),
        security: ids.includes('security'),
        taskEdit: ids.includes('taskEdit'),
        triggerEdit: ids.includes('triggerEdit'),
    };
};

export const countFiltersApplied = (filter: ActivityLogsFilterSettings): number => {
    let filtersApplied = 0;
    if (
        filter.other ||
        filter.comments ||
        filter.lifecycle ||
        filter.releaseEdit ||
        filter.triggerEdit ||
        filter.deliveryEdit ||
        filter.taskEdit ||
        filter.security ||
        filter.execution ||
        filter.reassign ||
        filter.reportingRecordEdit
    )
        filtersApplied++;
    if (filter.activityTypes.length > 0) filtersApplied++;
    if (filter.usernames.length > 0) filtersApplied++;
    if (filter.from) filtersApplied++;
    if (filter.to) filtersApplied++;
    return filtersApplied;
};

export const getTableColumns = (container: ActivityLogsContainerEnum) => {
    const timeZoneCity = getCityFromClientTimezone();
    const nameColumn = { id: 'name', label: 'Performed by', truncate: true, sortable: false, width: '20%' };
    const messageColumn = { id: 'message', label: 'Activity', sortable: false };
    const targetTypeColumn = { id: 'targetType', label: 'Target', sortable: false, width: '200px' };
    const detailsColumn = { id: 'details', label: 'Details', sortable: false, width: '150px' };
    const eventTimeColumn = { id: 'eventTime', label: `Time (${timeZoneCity})`, sortable: true, width: '150px' };

    if (container === ActivityLogsContainerEnum.TRIGGER) {
        return [nameColumn, messageColumn, detailsColumn, eventTimeColumn];
    } else {
        return [nameColumn, messageColumn, targetTypeColumn, eventTimeColumn];
    }
};

export const getActivityLogColumnMessage = (label: string, message: string): string => {
    const messageWithoutLabel = message.replace(label, '').trim();
    return `${messageWithoutLabel.charAt(0).toLowerCase()}${messageWithoutLabel.slice(1)}`;
};

export const calculateActivityLogColumnProps = (activityType: string, message: string): ActivityLogColumnProps | null => {
    const style = getActivityStyleAndLabel(activityType);

    if (!style) return null;

    return {
        background: style.background,
        border: style.border,
        label: style.label,
        message: getActivityLogColumnMessage(style.label, message),
    };
};

export const getTargetTypeLabel = (targetType: string, containerEnum: ActivityLogsContainerEnum): string => {
    if (containerEnum === ActivityLogsContainerEnum.RELEASE) {
        switch (targetType) {
            case 'xlrelease.Release':
                return 'Release';
            case 'xlrelease.Phase':
                return 'Phase';
            case 'xlrelease.Team':
                return 'Team';
            case 'xlrelease.StringVariable':
            case 'xlrelease.PasswordStringVariable':
            case 'xlrelease.BooleanVariable':
            case 'xlrelease.IntegerVariable':
            case 'xlrelease.ListStringVariable':
            case 'xlrelease.ReferenceVariable':
            case 'xlrelease.DateVariable':
            case 'xlrelease.MapStringStringVariable':
            case 'xlrelease.SetStringVariable':
                return 'Variable';
            case 'xlrelease.Attachment':
                return 'Attachment';
            case 'xlrelease.Comment':
                return 'Task';
            case 'xlrelease.TemplateLogo':
                return 'Template logo';
            case 'xlrelease.Folder':
                return 'Folder';
            default:
                return 'Task';
        }
    }

    switch (targetType) {
        case 'delivery.Delivery':
            return 'Delivery';
        case 'delivery.Stage':
            return 'Stage';
        case 'delivery.TrackedItem':
            return 'Tracked item';
        case 'delivery.StageTrackedItem':
            return 'Stage tracked item';
        case 'delivery.Transition':
            return 'Transition';
        case 'delivery.Condition':
        case 'delivery.ItemsCompletionCondition':
        case 'delivery.ConditionGroup':
        case 'delivery.TimeCondition':
            return 'Condition';
        default:
            return targetType;
    }
};

export const getTaskWithOpenedModalHref = (releaseId: string, phaseTaskId: string) => {
    return `${getReleaseHref(releaseId)}?openTaskDetailsModal=${phaseTaskId}`;
};

export const getReleaseHref = (releaseId: string) => {
    const releaseViewSegment = getPageTypeFromUrl();
    return `#/${releaseViewSegment}/${releaseId}`;
};

export const getTriggerHref = (triggerId: string) => {
    return `#/trigger-management/${triggerId.split('-').pop()}`;
};

export const getReleaseVariableHref = (releaseId: string, variableId: string) => {
    const releaseViewSegment = getPageTypeFromUrl();
    return `#/${releaseViewSegment}/${releaseId}/variables?variable=${variableId}`;
};

export const getDeliveryUrlFromPath = (deliveryId: string) => {
    const urlFromHash = window.location.hash;
    return urlFromHash.slice(0, urlFromHash.indexOf(deliveryId) + deliveryId.length);
};

export const removeAllAfterLastTask = (taskId: string) => {
    const regex = /(.*Task[^/-]*)/i;
    const match = regex.exec(taskId);
    return match ? match[1] : taskId;
};

export const checkIfImportantActivity = (activityType: string): boolean => IMPORTANT_ACTIVITY_TYPES.includes(activityType);
