import { SagaIterator } from 'redux-saga';
import { all, call, put, select, takeEvery } from 'redux-saga/effects';
import { remoteRunners, UpdateRemoteRunnerStatePayload } from './remote-runners.reducer';
import remoteRunnersSaga, {
    deleteRemoteRunnerAction,
    loadPermissionsAction,
    loadRemoteRunnersAction,
    reloadRemoteRunnersAction,
    updateRemoteRunnerStateAction,
    withLoadingState,
} from './remote-runners.saga';
import { httpDELETE, httpGET, httpPUT } from '@xlr-ui/app/features/common/services/http';
import { RemoteRunner } from '../remote-runners.types';
import { remoteRunnerMock } from '../__mocks__/remote-runners.mocks';
import { ConfigurationCallbackPayload } from '@xlr-ui/app/features/configuration/types';
import { PayloadAction } from '@reduxjs/toolkit';
import { getRemoteRunnersSelector } from './remote-runners.selectors';
import { hasEditRunnerPermissionRelease, hasViewRunnerPermissionRelease } from '@xlr-ui/app/js/auth/authenticator.saga';

const {
    loadRemoteRunners,
    loadPermissions,
    setRemoteRunners,
    reloadRemoteRunners,
    updateRemoteRunnerState,
    deleteRemoteRunner,
    setIsLoading,
    setEditRunners,
    setViewRunners,
} = remoteRunners.actions;

describe('remoteRunnersSaga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = remoteRunnersSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeEvery(loadRemoteRunners, loadRemoteRunnersAction),
                takeEvery(loadPermissions, loadPermissionsAction),
                takeEvery(reloadRemoteRunners, reloadRemoteRunnersAction),
                takeEvery(updateRemoteRunnerState, updateRemoteRunnerStateAction),
                takeEvery(deleteRemoteRunner, deleteRemoteRunnerAction),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('withLoadingState', () => {
        it('should handle success case of withLoadingState effect', () => {
            const effect = call(httpGET, 'url');
            const gen: SagaIterator = withLoadingState(effect);
            expect(gen.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(gen.next().value).toStrictEqual(effect);
            expect(gen.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(gen.next().done).toBe(true);
        });

        it('should handle finally case of withLoadingState effect', () => {
            try {
                const effect = call(httpGET, 'url');
                const gen: SagaIterator = withLoadingState(effect);
                expect(gen.next().value).toStrictEqual(put(setIsLoading(true)));
                expect(gen.throw?.(new Error('Test error')).value).toEqual(put(setIsLoading(false)));
                expect(gen.next().done).toBe(true);
            } catch (e) {
                //
            }
        });
    });

    describe('loadRemoteRunnersAction', () => {
        it('should handle loadRemoteRunners action', () => {
            const gen: SagaIterator = loadRemoteRunnersAction();
            expect(gen.next().value).toStrictEqual(
                call(withLoadingState, call(httpGET, `api/v1/config/byTypeAndTitle?configurationType=xlrelease.RemoteJobRunner`)),
            );
            expect(gen.next({ data: [remoteRunnerMock] }).value).toStrictEqual(put(setRemoteRunners([remoteRunnerMock])));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('loadPermissionsAction', () => {
        it('should handle loadPermissions action and set both permissions', () => {
            const gen: SagaIterator = loadPermissionsAction();
            expect(gen.next().value).toStrictEqual(call(hasEditRunnerPermissionRelease));
            expect(gen.next(true).value).toStrictEqual(call(hasViewRunnerPermissionRelease));
            expect(gen.next(true).value).toStrictEqual(put(setEditRunners(true)));
            expect(gen.next().value).toStrictEqual(put(setViewRunners(true)));
            expect(gen.next().done).toBeTruthy();
        });

        it('should handle loadPermissions action when user has no edit permission', () => {
            const gen: SagaIterator = loadPermissionsAction();
            expect(gen.next().value).toStrictEqual(call(hasEditRunnerPermissionRelease));
            expect(gen.next(false).value).toStrictEqual(call(hasViewRunnerPermissionRelease));
            expect(gen.next(true).value).toStrictEqual(put(setEditRunners(false)));
            expect(gen.next().value).toStrictEqual(put(setViewRunners(true)));
            expect(gen.next().done).toBeTruthy();
        });

        it('should handle loadPermissions action when user has no permissions', () => {
            const gen: SagaIterator = loadPermissionsAction();
            expect(gen.next().value).toStrictEqual(call(hasEditRunnerPermissionRelease));
            expect(gen.next(false).value).toStrictEqual(call(hasViewRunnerPermissionRelease));
            expect(gen.next(false).value).toStrictEqual(put(setEditRunners(false)));
            expect(gen.next().value).toStrictEqual(put(setViewRunners(false)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('reloadRemoteRunnersAction', () => {
        it('should handle reloadRemoteRunners action', () => {
            const action: PayloadAction<ConfigurationCallbackPayload> = {
                payload: {
                    id: '',
                    isUpdate: false,
                },
                type: '',
            };
            const gen: SagaIterator = reloadRemoteRunnersAction(action);
            expect(gen.next().value).toStrictEqual(call(loadRemoteRunnersAction));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('updateRemoteRunnerStateAction', () => {
        it('should handle updateRemoteRunnerState action', () => {
            const action: PayloadAction<UpdateRemoteRunnerStatePayload> = {
                type: '',
                payload: {
                    id: remoteRunnerMock.id,
                    enable: false,
                },
            };
            const updatedRunner = {
                ...remoteRunnerMock,
                enabled: false,
            };

            const gen: SagaIterator = updateRemoteRunnerStateAction(action);
            expect(gen.next().value).toStrictEqual(call(httpGET, `api/v1/config/${action.payload.id}`));
            expect(gen.next({ data: updatedRunner }).value).toStrictEqual(call(httpPUT, `api/v1/config/${action.payload.id}`, updatedRunner));

            const runners: RemoteRunner[] = [remoteRunnerMock];
            expect(gen.next({ data: runners }).value).toStrictEqual(select(getRemoteRunnersSelector));

            const updatedRunners: RemoteRunner[] = [updatedRunner];
            expect(gen.next(updatedRunners).value).toStrictEqual(put(setRemoteRunners(updatedRunners)));

            expect(gen.next().done).toBeTruthy();
        });

        it('should handle updateRemoteRunnerState action when runner ids match', () => {
            const action: PayloadAction<UpdateRemoteRunnerStatePayload> = {
                type: '',
                payload: {
                    id: remoteRunnerMock.id,
                    enable: false,
                },
            };
            const updatedRunner = {
                ...remoteRunnerMock,
                enabled: false,
            };

            const gen: SagaIterator = updateRemoteRunnerStateAction(action);
            expect(gen.next().value).toStrictEqual(call(httpGET, `api/v1/config/${action.payload.id}`));
            expect(gen.next({ data: updatedRunner }).value).toStrictEqual(call(httpPUT, `api/v1/config/${action.payload.id}`, updatedRunner));
            expect(gen.next({ data: updatedRunner }).value).toStrictEqual(select(getRemoteRunnersSelector));
            const updatedRunners: RemoteRunner[] = [updatedRunner];
            expect(gen.next(updatedRunners).value).toStrictEqual(put(setRemoteRunners(updatedRunners)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('deleteRemoteRunnerAction', () => {
        it('should handle deleteRemoteRunner action', () => {
            const action: PayloadAction<RemoteRunner> = {
                payload: remoteRunnerMock,
                type: '',
            };
            const gen: SagaIterator = deleteRemoteRunnerAction(action);
            expect(gen.next().value).toStrictEqual(call(httpDELETE, `api/v1/config/${action.payload.id}`));

            const remoteRunnersToSet = [{ ...remoteRunnerMock, id: '123' }];
            const runners: RemoteRunner[] = [remoteRunnerMock, ...remoteRunnersToSet];
            expect(gen.next().value).toStrictEqual(select(getRemoteRunnersSelector));
            expect(gen.next(runners).value).toStrictEqual(put(setRemoteRunners(remoteRunnersToSet)));

            expect(gen.next().done).toBeTruthy();
        });
    });
});
