import React from 'react';
import { DotClickAwayListener } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { ActivityLogsFilterDrawer, ActivityLogsFilterDrawerProps } from './activity-logs-filter-drawer.component';
import { FilterGroup } from '@xlr-ui/app/features/common/components/filter/filter-group.component';
import { FilterAutocomplete } from './filters/filter-autocomplete.component';
import { FilterDate } from '@xlr-ui/app/features/common/components/filter/filter-date.component';
import { FilterDrawer } from '@xlr-ui/app/features/common/components/filter-drawer/filter-drawer.component';
import { FilterUsername } from '@xlr-ui/app/features/common/components/filter/filter-username.component';
import { getActivityLogsCategories } from '../helper';
import { ActivityLogsContainerEnum } from '../types';
import { DEFAULT_FILTER_SETTINGS } from '../../../constants';

describe('ActivityLogsFilterDrawer component', () => {
    let wrapper: ReactWrapper;
    const onClearAll = jest.fn();
    const onDrawerClose = jest.fn();
    const onFilterChange = jest.fn();
    const numberOfFiltersApplied = 1;

    const defaultProps: ActivityLogsFilterDrawerProps = {
        activityTypes: [
            {
                id: '1',
                title: 'activityType1',
            },
        ],
        categories: getActivityLogsCategories(ActivityLogsContainerEnum.RELEASE),
        filter: {
            ...DEFAULT_FILTER_SETTINGS,
            activityTypes: ['1'],
            usernames: ['username1'],
            comments: true,
        },
        isDrawerOpened: true,
        numberOfFiltersApplied,
        onClearAll,
        onDrawerClose,
        onFilterChange,
        users: [
            {
                username: 'username1',
            },
        ],
    };

    const getDotClickAwayListener = () => wrapper.find(DotClickAwayListener);
    const getFilterDrawer = () => wrapper.find(FilterDrawer);
    const getFilterGroup = (title: string) => wrapper.findWhere((node) => node.is(FilterGroup) && node.prop('title') === title);

    const mount = (props: ActivityLogsFilterDrawerProps = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<ActivityLogsFilterDrawer {...props} />, jest.fn(), { profile: { dateFormat: 'MM/dd/yyyy', timeFormat: 'HH:mm:ss' } });
    };

    it('should render properly', () => {
        mount();
        expect(getDotClickAwayListener()).toExist();
        const filterDrawer = getFilterDrawer();
        expect(filterDrawer).toExist();
        const filterDrawerProps = filterDrawer.props();
        expect(filterDrawerProps.isOpened).toBe(true);
        expect(filterDrawerProps.numberOfFiltersApplied).toBe(numberOfFiltersApplied);
        expect(filterDrawerProps.onClearAllFilters).toStrictEqual(expect.any(Function));
        expect(filterDrawerProps.onClose).toStrictEqual(expect.any(Function));

        filterDrawer.invoke('onClose')?.();
        expect(onDrawerClose).toHaveBeenCalledTimes(1);
    });

    it('should clear all the filters', () => {
        mount();
        getFilterDrawer().invoke('onClearAllFilters')?.();
        expect(onClearAll).toHaveBeenCalledTimes(1);
    });

    it('should show all filters by default and handle change', () => {
        mount();
        expect(wrapper.find(FilterGroup)).toHaveLength(5);

        const activityProps = getFilterGroup('Activity').find(FilterAutocomplete).props();
        expect(activityProps.inputId).toStrictEqual('activityTypes');
        expect(activityProps.options).toStrictEqual(defaultProps.activityTypes);
        expect(activityProps.placeholder).toStrictEqual('Filter by activity type');
        expect(activityProps.value).toStrictEqual([{ id: '1', title: 'activityType1' }]);
        activityProps.onChange(['2']);
        expect(onFilterChange).toHaveBeenCalledWith({ ...defaultProps.filter, activityTypes: ['2'] });
        onFilterChange.mockReset();

        const usernamesProps = getFilterGroup('Performed by').find(FilterUsername).props();
        expect(usernamesProps.inputId).toStrictEqual('usernames');
        expect(usernamesProps.users).toStrictEqual(defaultProps.users);
        expect(usernamesProps.placeholder).toStrictEqual('Filter by user or system');
        expect(usernamesProps.value).toStrictEqual(['username1']);
        usernamesProps.onChange(['2']);
        expect(onFilterChange).toHaveBeenCalledWith({ ...defaultProps.filter, usernames: ['2'] });
        onFilterChange.mockReset();

        const categoriesProps = getFilterGroup('Category').find(FilterAutocomplete).props();
        expect(categoriesProps.className).toStrictEqual('categories-autocomplete');
        expect(categoriesProps.inputId).toStrictEqual('categories');
        expect(categoriesProps.options).toStrictEqual(defaultProps.categories);
        expect(categoriesProps.placeholder).toStrictEqual('Filter by category');
        expect(categoriesProps.value).toStrictEqual([
            {
                id: 'comments',
                title: 'Comments',
            },
        ]);
        categoriesProps.onChange(['releaseEdit', 'comments']);
        expect(onFilterChange).toHaveBeenCalledWith({ ...defaultProps.filter, comments: true, releaseEdit: true });
        onFilterChange.mockReset();

        const timeFromProps = getFilterGroup('Time from').find(FilterDate).props();
        expect(timeFromProps.inputId).toStrictEqual('time-from');
        expect(timeFromProps.value).toStrictEqual(defaultProps.filter.from);
        timeFromProps.onChange(123);
        expect(onFilterChange).toHaveBeenCalledWith({ ...defaultProps.filter, from: 123 });
        onFilterChange.mockReset();

        const timeToProps = getFilterGroup('Time to').find(FilterDate).props();
        expect(timeToProps.inputId).toStrictEqual('time-to');
        expect(timeToProps.value).toStrictEqual(defaultProps.filter.to);
        timeToProps.onChange(123);
        expect(onFilterChange).toHaveBeenCalledWith({ ...defaultProps.filter, to: 123 });
        onFilterChange.mockReset();
    });
});
