import React from 'react';
import { DotTable } from '@digital-ai/dot-components';
import { act, mockIntersectionObserver, mockResizeObserver, mountWithTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { ActivityLogsContainerEnum } from '../../types';
import { ActivityLogsTable, ActivityLogsTableProps } from './activity-logs-table.component';
import { LOGS_NUMBER_OF_SKELETONS } from '../../constants';
import { getTableColumns } from '../../helper';
import { ActivityLogsUser } from '../../../../components/activity-logs-user.component';
import { ActivityLogsActivity } from './activity-logs-activity.component';
import { ActivityLogsTypeDelivery } from './activity-logs-type.delivery.component';
import { ActivityLogsTypeRelease } from './activity-logs-type.release.component';
import { ActivityLogsTime } from './activity-logs-time.component';
import { ActivityLog } from '../../../../types';
import { getActivityLogUserTitle } from '../../../../helper';
import { mockReleaseActivityLogs } from '../../../../__mocks__/index.mock';

describe('ActivityLogsTable component', () => {
    let wrapper: ReactWrapper;
    const observe = jest.fn();
    const unobserve = jest.fn();
    const onIntersect = jest.fn();
    const onDetailsClick = jest.fn();
    const onUpdateTableData = jest.fn();
    const logs: ActivityLog[] = mockReleaseActivityLogs;

    const defaultProps: ActivityLogsTableProps = {
        dateOrder: 'asc',
        containerEnum: ActivityLogsContainerEnum.RELEASE,
        isFirstPage: true,
        isImportantHighlighted: false,
        isLastPage: false,
        isLoading: false,
        logs,
        onDetailsClick,
        onIntersect,
        onUpdateTableData,
        dateFormat: 'MM/dd/yyyy',
        timeFormat: 'HH:mm:ss',
        validTargetIds: [],
    };

    const getDotTable = () => wrapper.find(DotTable);
    const getActivityLogsUser = () => wrapper.find(ActivityLogsUser);
    const getActivityLogsActivity = () => wrapper.find(ActivityLogsActivity);
    const getActivityLogsTypeForDelivery = () => wrapper.find(ActivityLogsTypeDelivery);
    const getActivityLogsTypeForRelease = () => wrapper.find(ActivityLogsTypeRelease);
    const getActivityLogsTime = () => wrapper.find(ActivityLogsTime);
    const getImportantActivityRows = () => wrapper.find('tr.important-activity');

    const mount = async (props: ActivityLogsTableProps = defaultProps) => {
        await act(async () => {
            wrapper = mountWithTheme(<ActivityLogsTable {...props} />);
        });
    };

    beforeEach(() => {
        mockIntersectionObserver(observe, unobserve);
        mockResizeObserver();
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should render properly', async () => {
        await mount();

        const dotTable = getDotTable();
        expect(dotTable).toExist();
        const dotTableProps = dotTable.props();
        expect(dotTableProps.className).toBe('activity-logs-table');
        expect(dotTableProps.columns).toStrictEqual(getTableColumns(defaultProps.containerEnum));
        expect(dotTableProps.data).toStrictEqual(expect.any(Object));
        expect(dotTableProps.onUpdateData).toStrictEqual(expect.any(Function));
        expect(dotTableProps.order).toBe('asc');
        expect(dotTableProps.orderBy).toBe('eventTime');

        expect(getImportantActivityRows()).toHaveLength(0);
    });

    it('should render table with no data when no logs', async () => {
        await mount({ ...defaultProps, logs: [] });

        const dotTable = getDotTable();
        expect(dotTable).toExist();
        expect(dotTable.props().data).toStrictEqual([]);
    });

    it('should render table with proper components and data', async () => {
        const activityLog = {
            user: {
                username: 'admin',
                fullName: 'Release Administrator',
            },
            activityType: 'RELEASE_STARTED',
            message: 'Started the release',
            eventTime: 1726472112657,
            targetType: 'time.Schedule',
            targetId: 'Applications/Folder1/Release1',
            dataId: null,
        };
        await mount({ ...defaultProps, logs: [activityLog], isImportantHighlighted: true });
        const activityLogUserComponentProps = getActivityLogsUser().props();
        expect(activityLogUserComponentProps.user).toStrictEqual(activityLog.user);
        expect(activityLogUserComponentProps.title).toStrictEqual(getActivityLogUserTitle(activityLog.user));

        const activityLogActivityComponentProps = getActivityLogsActivity().props();
        expect(activityLogActivityComponentProps.activityType).toBe(activityLog.activityType);
        expect(activityLogActivityComponentProps.message).toBe(activityLog.message);

        const activityLogTypeComponentProps = getActivityLogsTypeForRelease().props();
        expect(activityLogTypeComponentProps.targetId).toBe(activityLog.targetId);
        expect(activityLogTypeComponentProps.targetType).toBe(activityLog.targetType);
        expect(activityLogTypeComponentProps.validTargetIds).toStrictEqual([]);

        const activityLogTimeComponentProps = getActivityLogsTime().props();
        expect(activityLogTimeComponentProps.dateFormat).toBe(defaultProps.dateFormat);
        expect(activityLogTimeComponentProps.eventTime).toBe(activityLog.eventTime);
        expect(activityLogTimeComponentProps.timeFormat).toBe(defaultProps.timeFormat);

        expect(getImportantActivityRows()).toHaveLength(1);
    });

    it('should render activity log type for Deliveries', async () => {
        const activityLog = {
            user: {
                username: 'admin',
                fullName: 'Release Administrator',
            },
            activityType: 'DELIVERY_CREATED',
            message: 'Delivery created',
            eventTime: 1726472112657,
            targetType: 'delivery.Delivery',
            targetId: 'Deliveries/Delivery123',
            dataId: null,
        };
        await mount({ ...defaultProps, logs: [activityLog], containerEnum: ActivityLogsContainerEnum.DELIVERY });
        const activityLogTypeComponentProps = getActivityLogsTypeForDelivery().props();
        expect(activityLogTypeComponentProps.targetId).toBe(activityLog.targetId);
        expect(activityLogTypeComponentProps.targetType).toBe(activityLog.targetType);
    });

    it('should not render activity log type for Triggers', async () => {
        const activityLog = {
            user: {
                username: 'admin',
                fullName: 'Release Administrator',
            },
            activityType: 'TRIGGER_CREATED',
            message: 'Trigger created',
            eventTime: 1726472112657,
            targetType: 'time.Schedule',
            targetId: 'Trigger',
            dataId: null,
        };
        await mount({ ...defaultProps, logs: [activityLog], containerEnum: ActivityLogsContainerEnum.TRIGGER });
        expect(getActivityLogsTypeForDelivery()).not.toExist();
        expect(getActivityLogsTypeForRelease()).not.toExist();
    });

    it('should render table with proper data when no logs and isLoading is set to true', async () => {
        await mount({ ...defaultProps, logs: [], isLoading: true });
        const dotTable = getDotTable();
        expect(dotTable).toExist();
        expect(dotTable.props().data.length).toBe(LOGS_NUMBER_OF_SKELETONS);
    });

    it('should execute correct handler when onUpdateData has been invoked', async () => {
        await mount();
        const dotTable = getDotTable();
        dotTable.invoke('onUpdateData')('asc');
        expect(onUpdateTableData).toHaveBeenCalledWith(true);
    });
});
