import React from 'react';
import { DotLink, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import IdsFactory from '@xlr-ui/app/js/util/ids';
import { getReleaseHref, getReleaseVariableHref, getTargetTypeLabel, getTaskWithOpenedModalHref, getTriggerHref, removeAllAfterLastTask } from '../../helper';
import { ActivityLogsContainerEnum } from '../../types';

export interface ActivityLogsTypeReleaseProps {
    targetId: string | null;
    targetType: string | null;
    validTargetIds: string[];
}

const idsFactory = IdsFactory();

export const ActivityLogsTypeRelease = ({ targetId, targetType, validTargetIds }: ActivityLogsTypeReleaseProps) => {
    if (!targetId || !targetType) {
        return (
            <DotTypography data-testid="type-label-typography" variant="body1">
                <DotTooltip
                    placement="left"
                    title="Link is unavailable because the target item cannot be identified or referenced for releases and templates older than version 9.0.0."
                >
                    <>Not available</>
                </DotTooltip>
            </DotTypography>
        );
    }

    const renderNotFound = (label: string) => (
        <DotTypography data-testid="type-label-typography" variant="body1">
            <DotTooltip placement="left" title="Link is unavailable because the target item has been deleted or moved.">
                <>{label}</>
            </DotTooltip>
        </DotTypography>
    );

    const renderLabel = (label: string) => (
        <DotTypography data-testid="type-label-typography" variant="body1">
            <>{label}</>
        </DotTypography>
    );

    const renderTaskNotIdentified = (label: string, targetTypeContext: string) => (
        <DotTypography data-testid="type-label-typography" variant="body1">
            <DotTooltip placement="left" title={`Link is unavailable because the target item cannot be identified or referenced (${targetTypeContext}).`}>
                <>{label}</>
            </DotTooltip>
        </DotTypography>
    );

    const internalId = idsFactory.toInternalId(targetId);

    // Handle Release type
    if (targetType === 'xlrelease.Release') {
        const releaseId = idsFactory.releaseIdFrom(internalId);
        return <DotLink href={getReleaseHref(releaseId)}>Release</DotLink>;
    }

    // Handle Variable type
    if (idsFactory.isVariableId(internalId)) {
        const releaseId = idsFactory.releaseIdFrom(internalId);
        return validTargetIds.includes(targetId) ? (
            <DotLink href={getReleaseVariableHref(releaseId, idsFactory.getName(targetId))}>Variable</DotLink>
        ) : (
            renderNotFound('Variable')
        );
    }

    // Handle Trigger type - ID-based detection for all trigger types
    if (idsFactory.isTriggerId(internalId)) {
        const triggerId = idsFactory.triggerIdFrom(internalId);
        return <DotLink href={getTriggerHref(triggerId)}>Trigger</DotLink>;
    }

    // Handle Task type
    const cleanTaskId = removeAllAfterLastTask(internalId);
    if (idsFactory.isTaskId(cleanTaskId)) {
        const label = getTargetTypeLabel(targetType, ActivityLogsContainerEnum.RELEASE);
        const taskExists = validTargetIds.includes(cleanTaskId);

        if (!taskExists) {
            return renderNotFound(label);
        }

        const releaseId = idsFactory.releaseIdFrom(internalId);
        const phaseTaskId = idsFactory.phaseTaskIdFrom(internalId);
        const cleanPhaseTaskId = removeAllAfterLastTask(phaseTaskId);
        return <DotLink href={getTaskWithOpenedModalHref(releaseId, cleanPhaseTaskId)}>{label}</DotLink>;
    }

    // Handle other types
    const label = getTargetTypeLabel(targetType, ActivityLogsContainerEnum.RELEASE);
    return label === 'Task' ? renderTaskNotIdentified(label, targetType) : renderLabel(label);
};
