import React, { useEffect, useState } from 'react';

import { StageListHeaderComponent } from './stage-list-header-component';
import { StageListItemComponent } from './stage-list-item-component';
import { StageEditModal } from './stage-edit-modal';
import { AlertMessage, alertType, HelpBlock, NoData } from '@xlr-ui/app/react/components';

import './../../../styles/common.less';
import { Stage } from '../../../application-list.types';
import { DotConfirmationDialog, DotThemeProvider, DotIcon } from '@digital-ai/dot-components';
import { stageState } from '../ducks/stage.reducer';
import { getStageList } from '../ducks/environment-list.selectors';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';

const { searchStages, deleteStage, saveStage } = stageState.actions;

const helpMessage = `Use Stages to logically group environments. You can also use Stages to filter the environment scheduler and dashboards.`;
export const StageListComponent = () => {
    const dispatch = useAppDispatch();
    const stageList: Array<Stage> = useAppSelector(getStageList);
    const [deleteModalIsOpen, setDeleteModalIsOpen] = useState(false);
    const [editModalIsOpen, setEditModalIsOpen] = useState(false);
    const [stage, setStage] = useState({
        id: '',
        title: '',
    } as Stage);

    useEffect(() => {
        dispatch(searchStages(''));
    }, []);

    const onFilterChange = (text: string) => {
        dispatch(searchStages(text));
    };

    const showDeleteModal = (st: Stage) => {
        setDeleteModalIsOpen(true);
        setStage(st);
    };

    const showEditModal = (st: Stage) => {
        setEditModalIsOpen(true);
        setStage(st);
    };

    const onCloseDeleteModal = (action: string) => {
        if (action === 'delete') {
            dispatch(deleteStage(stage));
        }
        setDeleteModalIsOpen(false);
        setStage({
            id: '',
            title: '',
        });
    };

    const handleStageSave = (st: Stage) => {
        dispatch(saveStage(st));
        setEditModalIsOpen(false);
        setStage({
            id: '',
            title: '',
        });
    };

    const handleCancel = () => {
        setEditModalIsOpen(false);
        setStage({
            id: '',
            title: '',
        });
    };

    const renderDeleteModal = () => {
        return (
            <DotConfirmationDialog
                message={
                    <div>
                        <AlertMessage alertType={alertType.WARNING}>
                            <p>Deleting a stage can lead to missing data for connected resources on reporting and environment scheduling.</p>
                        </AlertMessage>
                        <p>
                            You are about to delete stage <strong>{stage.title}</strong>.
                        </p>
                    </div>
                }
                onCancel={() => onCloseDeleteModal('cancel')}
                onSubmit={() => onCloseDeleteModal('delete')}
                open={deleteModalIsOpen}
                submitButtonProps={{ className: 'submitBtn', label: 'Delete Stage', startIcon: <DotIcon iconId="delete" />, type: 'destructive' }}
                title="Delete stage"
            />
        );
    };

    const renderStage = (stage1: Stage) => {
        return <StageListItemComponent key={stage1.id} onDelete={showDeleteModal} onEdit={showEditModal} stage={stage1} />;
    };
    const renderStages = () => {
        const stages = stageList.map(renderStage);

        return (
            <div className="margin">
                <h4 className="page-title">Stages ({stages.length})</h4>
                <HelpBlock className="help-message-container">{helpMessage}</HelpBlock>
                {stages.length ? stages : <NoData className="no-stages-message" message="No stages found." />}
            </div>
        );
    };

    return (
        <DotThemeProvider>
            {renderDeleteModal()}
            <StageEditModal isOpened={editModalIsOpen} model={stage} onCancel={handleCancel} onSave={handleStageSave} />
            <StageListHeaderComponent onFilterChange={onFilterChange} />
            <div className="header-shadow"></div>
            {renderStages()}
        </DotThemeProvider>
    );
};
