import React from 'react';
import { DotIllustration } from '@digital-ai/dot-components';
import { act, mountWithTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { TextAreaMarkdown } from '@xlr-ui/app/react/components/markdown/textarea-markdown.component';
import { TileButtonsRow } from './tile-buttons-row.component';
import { primaryButton, secondaryButton, tertiaryButton } from './tile-buttons-row.mock';
import { CommonCardTileComponent, CommonCardTileComponentProps, MAX_DESCRIPTION_SIZE, MAX_DESCRIPTION_SIZE_WARNING } from './common-card-tile.component';
import { CommonTileHeader } from './common-tile-header.component';

describe('CommonCardTileComponent', () => {
    let wrapper: ReactWrapper;
    const description = 'my description';
    const isMarkdown = true;
    const title = 'title';
    const subtitle = 'subtitle';
    const imageLink = 'image.png';

    const defaultProps: CommonCardTileComponentProps = { description, isMarkdown, title, primaryButton };

    const mount = async (props: CommonCardTileComponentProps = defaultProps) => {
        await act(async () => {
            wrapper = mountWithTheme(<CommonCardTileComponent {...props} />);
        });
    };

    const getCardHeader = () => wrapper.find(CommonTileHeader);
    const getMarkdownDescription = () => wrapper.find(TextAreaMarkdown);
    const getImage = () => wrapper.find('.card-tile-content img[alt="tile image"]');
    const getIllustration = () => wrapper.find(DotIllustration);
    const getTileButtonsRow = () => wrapper.find(TileButtonsRow);

    describe('description', () => {
        it('should render description in markdown', async () => {
            await mount();
            const markdownProps = getMarkdownDescription().props();
            expect(markdownProps.text).toBe(description);
            expect(markdownProps.isReadOnly).toBe(true);
            expect(markdownProps.rows).toBe(5);
        });

        it('should render description in text', async () => {
            await mount({ ...defaultProps, isMarkdown: false });
            expect(getMarkdownDescription()).not.toExist();
            expect(wrapper).toIncludeText(description);
        });

        it('should render warning if text is too long', async () => {
            const fakeDescription = {
                length: MAX_DESCRIPTION_SIZE + 1,
            };
            await mount({ ...defaultProps, description: fakeDescription as never });
            const markdownProps = getMarkdownDescription().props();
            expect(markdownProps.text).toBe(MAX_DESCRIPTION_SIZE_WARNING);
        });
    });

    describe('tile header', () => {
        it('should render properly', async () => {
            await mount();
            expect(getCardHeader().props().title).toBe(title);
            expect(getCardHeader().props().avatarIcon).toBe(undefined);
            expect(getCardHeader().props().subtitle).toBe(undefined);
        });

        it('should render subtitle', async () => {
            await mount({ ...defaultProps, subtitle });
            expect(getCardHeader().props().subtitle).toBe(subtitle);
        });
    });

    describe('image', () => {
        it('it should render the image if imageLink is present', async () => {
            await mount({ ...defaultProps, imageLink });
            const image = getImage();
            expect(image).toExist();
            expect(image).toHaveProp('src', 'image.png');
        });

        it('it should not render the image if imageLink is not present', async () => {
            await mount();
            expect(getImage()).not.toExist();
        });
    });

    describe('illustration', () => {
        const illustrationId = 'test-illustration';

        it('it should render the illustration if illustrationId is present', async () => {
            await mount({ ...defaultProps, illustrationId });
            const illustration = getIllustration();
            expect(illustration).toExist();
            expect(illustration.props().illustrationId).toBe(illustrationId);
        });

        it('it should not render the illustration if illustrationId is not present', async () => {
            await mount();
            expect(getIllustration()).not.toExist();
        });

        it('it should render the illustration if both illustrationId and imageLink are present', async () => {
            await mount({ ...defaultProps, illustrationId, imageLink });
            expect(getIllustration()).toExist();
            expect(getImage()).not.toExist();
        });
    });

    describe('tile buttons row', () => {
        it('should render tile buttons row properly', async () => {
            await mount();
            const tileButtonsRowProps = getTileButtonsRow().props();
            expect(tileButtonsRowProps.primaryButton).toBe(primaryButton);
            expect(tileButtonsRowProps.secondaryButton).toBe(undefined);
            expect(tileButtonsRowProps.tertiaryButton).toBe(undefined);
            expect(tileButtonsRowProps.alignButtonsRight).toBe(undefined);
        });

        it('should render secondary button', async () => {
            await mount({ ...defaultProps, secondaryButton });
            expect(getTileButtonsRow().props().secondaryButton).toBe(secondaryButton);
        });

        it('should render tertiary button', async () => {
            await mount({ ...defaultProps, tertiaryButton });
            expect(getTileButtonsRow().props().tertiaryButton).toBe(tertiaryButton);
        });
    });
});
