import { SagaIterator } from 'redux-saga';
import { environmentLists } from './environment-list.reducer';
import environmentListSaga, {
    onAddEnvironment,
    onDeleteEnvironment,
    onEditEnvironment,
    onInitEnvironmentList,
    onSearch,
    toastr,
} from './environment-list.saga';
import { all, call, put, select, takeLatest } from 'redux-saga/effects';
import { httpDELETE, httpPOST } from '@xlr-ui/app/features/common/services/http';
import { envList, envList2, envWithId, stages } from '../__mocks__/environment-list.mocks';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import { getEnvironments } from './environment-list.selectors';

const { initEnvironmentList, searchEnvironments, createEnvironment, editEnvironment, deleteEnvironment, setEnvironments, setLoading, setStages } =
    environmentLists.actions;

describe('environment-list.saga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = environmentListSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(createEnvironment, onAddEnvironment),
                takeLatest(editEnvironment, onEditEnvironment),
                takeLatest(deleteEnvironment, onDeleteEnvironment),
                takeLatest(searchEnvironments, onSearch),
                takeLatest(initEnvironmentList, onInitEnvironmentList),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('search environments', () => {
        it('should search and set environments with stage', () => {
            const gen: SagaIterator = onSearch({
                payload: {
                    title: '',
                    stageTitle: 'as',
                    folderId: '',
                },
            });
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/search', { title: '', stage: 'as', folderId: '' }));
            expect(gen.next({ data: envList }).value).toStrictEqual(put(setEnvironments(envList)));
            expect(gen.next().done).toBe(true);
        });

        it('should search and set environments without stage', () => {
            const gen: SagaIterator = onSearch({
                payload: {
                    title: '',
                    stageTitle: '',
                    folderId: '',
                },
            });
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/search', { title: '', stage: null, folderId: '' }));
            expect(gen.next({ data: envList }).value).toStrictEqual(put(setEnvironments(envList)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('on add environment', () => {
        it('should open create environment page with no folder', () => {
            const gen: SagaIterator = onAddEnvironment({
                payload: '',
                type: 'whatever',
            });
            const $state = { go: jest.fn() };
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual($state.go('environmentCreate'));
            expect(gen.next().done).toBe(true);
        });

        it('should open create environment page with folder', () => {
            const gen: SagaIterator = onAddEnvironment({
                payload: 'Folder42',
                type: 'whatever',
            });
            const $state = { go: jest.fn() };
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual($state.go('folders.detail.environmentCreate'));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('on edit environment', () => {
        it('should go to the edit environment page with no folder', () => {
            const envId = 'someEnv';
            const gen: SagaIterator = onEditEnvironment({
                payload: {
                    envId,
                    folderId: '',
                },
                type: 'whatever',
            });
            const $state = { go: jest.fn() };
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual($state.go('environmentCreate', { environmentId: envId }));
            expect(gen.next().done).toBe(true);
        });

        it('should go to the edit environment page with folder', () => {
            const envId = 'someEnv';
            const gen: SagaIterator = onEditEnvironment({
                payload: {
                    envId,
                    folderId: 'Folder42',
                },
                type: 'whatever',
            });
            const $state = { go: jest.fn() };
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual($state.go('folders.detail.environmentCreate', { environmentId: envId }));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('on delete environment', () => {
        it('should delete environment', () => {
            const gen: SagaIterator = onDeleteEnvironment({
                payload: envWithId,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(put(setLoading(true)));
            expect(gen.next().value).toStrictEqual(call(httpDELETE, `api/v1/environments/${envWithId.id}`));
            expect(gen.next().value).toStrictEqual(put(setLoading(false)));
            expect(gen.next().value).toStrictEqual(select(getEnvironments));
            expect(gen.next(envList2).value).toStrictEqual(put(setEnvironments(envList)));
            expect(gen.next().value).toStrictEqual(call(toastr.success, `Environment [${envWithId.title}] deleted.`));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('on init environment list', () => {
        it('should initialize environment list state', () => {
            const gen: SagaIterator = onInitEnvironmentList({
                payload: '',
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/search', { title: '', stage: null, folderId: '' }));
            expect(gen.next({ data: envList }).value).toStrictEqual(call(httpPOST, 'api/v1/environments/stages/search', { title: '' }));
            expect(gen.next({ data: stages }).value).toStrictEqual(put(setEnvironments(envList)));
            expect(gen.next().value).toStrictEqual(put(setStages(stages)));
            expect(gen.next().done).toBe(true);
        });
    });
});
