package com.xebialabs.xlrelease.repository;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.springframework.util.StringUtils;

import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlrelease.domain.Phase;

/**
 * Used for restarting phases and indicate which 'version' of the phase should be restarted. For example: LATEST means that restarting should be done from the most recent phase. This
 * is determined by checking if the Phase has been copied or not (ie, {@link Phase#isPhaseCopied()}).
 *
 */
@PublicApiRef
public enum PhaseVersion {
    // See also internal API com.xebialabs.xlrelease.views.PhaseFullView, and its view builder that sets the expected values com.xebialabs.xlrelease.views.converters.PhaseViewConverter#toFullView(Phase, List)
    /**
     * Copy the most recent version of the phase. Determined by checking if the phase has not been copied yet (ie, there are no other phases pointing to this phase). See also {@link Phase#isPhaseCopied()}
     */
    LATEST,
    /**
     * Copy the original (first) version of the phase. Determined by checking if the phase origin is null.
     */
    ORIGINAL,
    /**
     * This is the 'old' behavior. Copy ALL the phases between 2 given phases. Regardless of versioning.
     */
    ALL;

    /**
     * Used by JAX-RS binding
     *
     * The input is being uppercased to allow for case insensitivity
     * @param s
     * @return
     */
    public static PhaseVersion fromString(final String s) {
        if (StringUtils.hasText(s)) {
            return PhaseVersion.valueOf(s.toUpperCase());
        }
        return PhaseVersion.ALL; // always fall back to ALL
    }

    /**
     * Given a phase, determine its phaseVersion(s). There can be more than one, in the case that a phase is the original one and also the most recent.
     *
     * @param phase
     * @return
     */
    public static List<PhaseVersion> determinePhaseVersion(final Phase phase) {
        if (phase == null) return Arrays.asList(PhaseVersion.ALL);

        List<PhaseVersion> phaseVersions = new ArrayList<>(2);
        if (phase.isOriginal()) phaseVersions.add(PhaseVersion.ORIGINAL);
        if (!phase.isPhaseCopied()) phaseVersions.add(PhaseVersion.LATEST);
        if (!phaseVersions.isEmpty()) return phaseVersions;

        return Arrays.asList(PhaseVersion.ALL);
    }

    /**
     * Determines the phaseVersion of the given phase and compares it with the given phaseVersion parameter.
     *
     * @param phase
     * @param phaseVersion
     * @return
     */
    public static boolean isPhaseVersion(final Phase phase, PhaseVersion phaseVersion) {
        return determinePhaseVersion(phase).contains(phaseVersion);
    }
}
