package com.xebialabs.xlrelease.plugin.manager.validator

import com.xebialabs.deployit.plugin.api.reflect.{InputHint, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.{Metadata, Property}

import java.lang.annotation.Annotation


sealed trait Change {
  //def isCompatible: Boolean = false
}

sealed trait TypeChange extends Change {
  def oldXlType: Option[Type]
  def newXlType: Option[Type]

  def xlTypeName: String
}

sealed trait AttributeChange[T] extends Change {
  def name: String
  def oldValue: T
  def newValue: T
  def cosmetic: Boolean = false

  override def toString: String = {
    s"Attribute [$name] changed from [$oldValue] to [$newValue]"
  }
}

case class NewType(newXlType: Option[Type]) extends TypeChange {
  override def oldXlType: Option[Type] = None

  override def xlTypeName: String = newXlType.get.toString
}

case class DeletedType(oldXlType: Option[Type]) extends TypeChange {
  override def newXlType: Option[Type] = None

  override def xlTypeName: String = oldXlType.get.toString
}

case class UpdatedType(oldXlType: Option[Type], newXlType: Option[Type], propertyChanges: Seq[PropertyChange], attributeChanges: Seq[AttributeChange[_]])
  extends TypeChange {
  override def xlTypeName: String = oldXlType.get.toString
}

sealed trait PropertyChange extends Change {
  def propertyName: String
}

case class NewProperty(propertyName: String) extends PropertyChange

case class DeletedProperty(propertyName: String) extends PropertyChange

case class UpdatedProperty(propertyName: String, attributeChanges: Seq[AttributeChange[_]]) extends PropertyChange

case class DescriptionChange(oldValue: String, newValue: String) extends AttributeChange[String] {
  override def name = "description"
  override def cosmetic = true
}

case class AsContainmentChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name = "as-containment"
}

case class NestedChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name = "nested"
}

case class CategoryChange(oldValue: String, newValue: String) extends AttributeChange[String] {
  override def name = "category"
  override def cosmetic = true
}

case class LabelChange(oldValue: String, newValue: String) extends AttributeChange[String] {
  override def name = "label"
  override def cosmetic = true
}

case class PasswordChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name = "password"
}

case class RequiredChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name = "required"
}

case class SizeChange(oldValue: Property.Size, newValue: Property.Size) extends AttributeChange[Property.Size] {
  override def name: String = "size"
  override def cosmetic = true
}

case class KindChange(oldValue: PropertyKind, newValue: PropertyKind) extends AttributeChange[PropertyKind] {
  override def name = "kind"
}

case class EnumValuesChange(oldValue: Seq[String], newValue: Seq[String]) extends AttributeChange[Seq[String]] {
  override def name = "enum-values"
}

case class ReferencedTypeChange(oldValue: Type, newValue: Type) extends AttributeChange[Type] {
  override def name = "referenced-type"
}

case class DefaultChange(oldValue: Object, newValue: Object) extends AttributeChange[Object] {
  override def name = "default"
}

case class HiddenChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name = "hidden"
}

case class TransientChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name = "transient"
}

case class ReadOnlyChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name = "readonly"
}

case class CandidateValuesFilter(oldValue: String, newValue: String) extends AttributeChange[String] {
  override def name = "candidate-values-filter"
}

case class AliasesChange(oldValue: Seq[String], newValue: Seq[String]) extends AttributeChange[Seq[String]] {
  override def name = "aliases"
}

case class InputHintChange(oldValue: InputHint, newValue: InputHint) extends AttributeChange[InputHint] {
  override def name: String = "inputHint"
  override def cosmetic = true

  override def toString: String = {
    s"Attribute [$name] changed from [${toString(oldValue)}] to [${toString(newValue)}]"
  }
  private def toString(source : InputHint): String =
    s"""
      |kind=${source.getKind}
      |required=${source.isRequired}
      |prompt=${source.getPrompt}
      |values=${source.getValues}
      |validationRules=${source.getValidationRules}
      |copyFromProperty=${source.getCopyFromProperty}
      |referencedType=${source.getReferencedType}
      |methodRef=${source.getMethodRef}
      |dynamicLookup=${source.isDynamicLookup}
      |""".stripMargin
}

case class AnnotationsChange(oldValue: Seq[Annotation], newValue: Seq[Annotation]) extends AttributeChange[Seq[Annotation]] {
  override def name = "aliases"
}

case class VirtualChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name: String = "virtual"
}

case class InterfacesChange(oldValue: Set[String], newValue: Set[String]) extends AttributeChange[Set[String]] {
  override def name: String = "interfaces"
}

case class SuperClassesChange(oldValue: Set[String], newValue: Set[String]) extends AttributeChange[Set[String]] {
  override def name: String = "superclasses"
}

case class VersionedChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name: String = "versioned"
}

case class ContainerTypeChange(oldValue: Type, newValue: Type) extends AttributeChange[Type] {
  override def name: String = "container-type"
}

case class DeployableTypeChange(oldValue: Type, newValue: Type) extends AttributeChange[Type] {
  override def name: String = "deployable-type"
}

case class InspectableChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name: String = "inspectable"
}

case class InspectionPropertyChange(oldValue: Boolean, newValue: Boolean) extends AttributeChange[Boolean] {
  override def name: String = "inspectionProperty"
}

case class RootChange(oldValue: Metadata.ConfigurationItemRoot, newValue: Metadata.ConfigurationItemRoot) extends AttributeChange[Metadata.ConfigurationItemRoot] {
  override def name: String = "root"
}

case class RootNameChange(oldValue: String, newValue: String) extends AttributeChange[String] {
  override def name: String = "rootName"
}