package com.xebialabs.xlrelease.booter

import com.xebialabs.deployit.booter.local.{LocalDescriptor, LocalDescriptorRegistry, SealedDescriptorHierarchy, Verifications}
import com.xebialabs.deployit.plugin.api.reflect.{BaseDescriptorRegistry, Descriptor, IDescriptorRegistry, Type}

import java.util
import java.util.Collections
import java.util.concurrent.ConcurrentHashMap
import scala.jdk.CollectionConverters._

class HierarchicalDescriptorRegistry(registryId: HierarchicalDescriptorRegistryId, val parentDescriptorRegistry: IDescriptorRegistry)
  extends BaseDescriptorRegistry(registryId) {

  private val parentDescriptors = parentDescriptorRegistry.getDescriptors()

  private val subtypes = new SealedDescriptorHierarchy()
  private val descriptors: util.Map[Type, Descriptor] = new ConcurrentHashMap[Type, Descriptor]()

  // we cannot add 2nd local registry
  override def isLocal: Boolean = false

  override def isDefault: Boolean = true

  // Make sure this registry is always given priority over the default local registry
  override def getOrder: Int = LocalDescriptorRegistry.ORDER - 1

  override def getDescriptors(): util.Collection[Descriptor] = {
    val r = new util.ArrayList[Descriptor]()
    r.addAll(parentDescriptors)
    r.addAll(descriptors.values())
    Collections.unmodifiableCollection(r)
  }

  override def getSubtypes(supertype: Type): util.Collection[Type] = {
    val r = new util.ArrayList[Type]()
    // there is no need to fetch subtypes from a parent registry as those will be registered into subtypes map via TypeDefinitions
    r.addAll(subtypes.getSubtypes(supertype))
    Collections.unmodifiableCollection(r)
  }

  override def getDescriptor(`type`: Type): Descriptor = {
    val our = descriptors.get(`type`)
    if (our != null) {
      our
    } else {
      parentDescriptorRegistry.getDescriptor(`type`)
    }
  }

  override def exists(`type`: Type): Boolean = {
    descriptors.containsKey(`type`) || parentDescriptorRegistry.exists(`type`)
  }

  override def register(descriptor: Descriptor): Unit = {
    // TypeDefinitions will not register an existing descriptor (i.e. one from parent descriptor registry), so we don't have to guard this method
    descriptors.put(descriptor.getType, descriptor)
  }

  override def registerSubtype(supertype: Type, subtype: Type): Unit = {
    // TypeDefinitions will register also descriptors that are in parent descriptor registry so we will have full type hierarchy
    subtypes.registerSubType(supertype, subtype)
  }

  override def verifyTypes(): Unit = {
    // make sure that types are valid (same way as LocalBooter does)
    parentDescriptorRegistry.verifyTypes()
    val verifications = new Verifications()
    descriptors.values().asScala.filterNot(_.isVirtual).foreach(_.asInstanceOf[LocalDescriptor].verify(verifications))
    verifications.done()
  }
}
