package com.xebialabs.xlrelease.plugin.manager.validator

import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, IDescriptorRegistry}
import com.xebialabs.plugin.manager.validator.{RestartNotNeeded, TypeSystemValidationResults, UnableToCreateTypeSystem}
import com.xebialabs.xlrelease.booter.XlrBooter
import com.xebialabs.xlrelease.plugin.ZipPlugin
import grizzled.slf4j.Logging

import scala.util.Using

class ZipPluginValidator(xlrBooter: XlrBooter,
                         typeChangeDetector: TypeChangeDetector,
                         typeChangeValidator: TypeChangeValidator) extends XlrPluginValidator with Logging {

  import scala.jdk.CollectionConverters._

  override def validate: Validate = {
    case plugin: ZipPlugin => validateZipPlugin(plugin)
  }

  private def validateZipPlugin(zipPlugin: ZipPlugin): TypeSystemValidationResults = {
    val hasClassFiles = zipPlugin.find("**/*.class").nonEmpty
    val hasTypeDefinitions = zipPlugin.find("type-definitions.*ml").exists { entry =>
      val filePath = entry.toString
      filePath.endsWith(".xml") || filePath.endsWith(".yaml")
    }
    if (hasClassFiles) {
      // jar plugins can contain classes but require a restart
      UnableToCreateTypeSystem(s"Plugin $zipPlugin contains class files", Seq.empty, Seq.empty)
    } else if (hasTypeDefinitions) {
      val bootResult = xlrBooter.bootWithTempRegistry(zipPlugin)
      // at this point registry should contain our new types - we cannot use Type.valueOf, but we can use Registry
      val firstPassResults = Option(bootResult.validationErrors)
        .filter(_.nonEmpty)
        .map(errors => UnableToCreateTypeSystem("Errors found attempting to start type system with the plugin", Seq.empty, errors))
        .getOrElse(TypeSystemValidationResults())

      val secondPassResults = Using.resource(bootResult.registry) { r =>
        validateTypeSystem(zipPlugin, r)
      }

      firstPassResults ++ secondPassResults
    } else {
      UnableToCreateTypeSystem("No type-definitions found", Seq.empty, Seq.empty)
    }
  }


  private def validateTypeSystem(zipPlugin: ZipPlugin, newRegistry: IDescriptorRegistry): TypeSystemValidationResults = {
    val newDescriptors = newRegistry.getDescriptors().asScala.toSeq
    val oldDescriptors = DescriptorRegistry.getDescriptors.asScala.toSeq

    try {
      val changes = typeChangeDetector.diff(oldDescriptors, newDescriptors, XlrBooter.RELEASE_REGISTRY_ID, newRegistry.getId)
      val results = typeChangeValidator.validate(changes)

      if (results.errors.nonEmpty) {
        val msg = "Type system validation failed for the plugin, see errors and warnings for more information."
        UnableToCreateTypeSystem(msg, results.warnings, results.errors)
      } else {
        RestartNotNeeded(zipPlugin.zipFile.getName, results.warnings)
      }
    } catch {
      case t: Throwable =>
        logger.error(s"Exception caught during validation: $t")
        UnableToCreateTypeSystem(t.toString, Seq.empty, Seq.empty)
    }
  }

}
