package com.xebialabs.xlrelease.plugin.manager.validator

import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.validator._
import com.xebialabs.plugin.manager.{Plugin => PlatformPlugin}
import com.xebialabs.xlrelease.plugin.{JarPlugin, ZipPlugin}
import grizzled.slf4j.Logging
import org.springframework.util.FileSystemUtils

import java.io.File
import java.nio.file.{Files, Path}
import scala.util.Using

class XlrTypeSystemValidator(xlrPluginValidators: Array[XlrPluginValidator]) extends TypeSystemValidator with Logging {
  override def validate(plugin: PlatformPlugin): TypeSystemValidationResults = {
    withTmpDir { tmpDir =>
      val pluginExtension = plugin.id.extension
      val maybeXlrPlugin = pluginExtension match {
        case ConfigWrapper.EXTENSION_JAR =>
          val file = getTemporaryPluginFile(tmpDir, plugin)
          Some(JarPlugin(file))
        case ConfigWrapper.EXTENSION_ZIP =>
          val file = getTemporaryPluginFile(tmpDir, plugin)
          Some(ZipPlugin(file))
        case _ =>
          None
      }
      maybeXlrPlugin match {
        case Some(xlrPlugin) =>
          Using.resource(xlrPlugin) { p =>
            xlrPluginValidators.map(_.validate).reduceLeft(_ orElse _)(p)
          }
        case None =>
          UnsupportedPluginType(s"$pluginExtension is not a supported plugin type", Seq.empty, Seq.empty)
      }
    }
  }

  private def getTemporaryPluginFile(tmpDir: Path, plugin: PlatformPlugin): File = {
    val pluginPath = tmpDir.resolve(plugin.id.filename())
    Files.write(pluginPath, plugin.bytes).toFile
  }

  private def withTmpDir[R](block: Path => R): R = {
    val tmpDir = Files.createTempDirectory("plugin")
    try {
      block(tmpDir)
    } finally {
      try {
        FileSystemUtils.deleteRecursively(tmpDir)
      } catch {
        case ex: Exception =>
          logger.error(s"Unable to delete ${tmpDir.toFile.getName}", ex)
          throw ex
      }
    }
  }
}
