package com.xebialabs.xlrelease.rules;

import java.util.*;
import org.junit.rules.ExternalResource;
import org.springframework.security.authentication.TestingAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.context.SecurityContextImpl;

import com.xebialabs.deployit.security.PermissionEditor;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.security.permission.PermissionHandler;

import static com.xebialabs.deployit.security.PermissionEnforcer.ROLE_ADMIN;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static java.util.Collections.addAll;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;

public class LoginRule extends ExternalResource {
    private static final String GLOBAL_SECURITY_ALIAS = "global";
    private static final String DEFAULT_PASSWORD = "password";

    private final String username;

    private final String password;

    private List<String> authorities;

    private LoginRule(String username, String password, String... authorities) {
        this.username = username;
        this.password = password;
        this.authorities = new ArrayList<>();
        addAll(this.authorities, authorities);
    }

    public static LoginRule loginWith(String username) {
        return new LoginRule(username, DEFAULT_PASSWORD);
    }

    public static LoginRule loginWithRoleAdmin(String username) {
        return new LoginRule(username, DEFAULT_PASSWORD, ROLE_ADMIN);
    }

    public void logout() {
        Permissions.clearSecurityContext();
    }

    @Override
    public void before() {
        SecurityContextImpl context = new SecurityContextImpl();
        context.setAuthentication(this.getAuthentication());
        SecurityContextHolder.setContext(context);
    }

    public Authentication getAuthentication() {
        return new TestingAuthenticationToken(username, password, authorities.toArray(new String[authorities.size()]));
    }

    @Override
    public void after() {
        Permissions.clearSecurityContext();
    }

    public static void grantAdminPermissionTo(String username, PermissionEditor permissionEditor, RoleService roleService) {
        Role roleWithAdminPermission = new Role(String.valueOf(roleService.getRoles().size()), "RoleWithAdminPermission");
        roleWithAdminPermission.getPrincipals().add(username);
        Map<Role, Set<Permission>> rolePermission = new HashMap<>();
        rolePermission.put(roleWithAdminPermission, singleton(ADMIN));
        roleService.writeRoleAssignments(singletonList(roleWithAdminPermission));
        permissionEditor.editPermissions(GLOBAL_SECURITY_ALIAS, rolePermission);
    }

    public static void grantGlobalPermissionsTo(String username, Set<Permission> permissions, PermissionEditor permissionEditor, RoleService roleService) {
        permissions.forEach(permission -> {
            assert permission.getLevel() == PermissionHandler.Level.GLOBAL || permission.getLevel() == PermissionHandler.Level.BOTH;
        });
        Role roleWithPermission = new Role(String.valueOf(roleService.getRoles().size()), "TestRoleForUser" + username);
        roleWithPermission.getPrincipals().add(username);
        Map<Role, Set<Permission>> rolePermission = new HashMap<>();
        rolePermission.put(roleWithPermission, permissions);
        roleService.writeRoleAssignments(singletonList(roleWithPermission));
        permissionEditor.editPermissions(GLOBAL_SECURITY_ALIAS, rolePermission);
    }

    public static void addUserToRole(String username, String roleName, RoleService roleService) {
        Role role = new Role(null, roleName);
        role.getPrincipals().add(username);
        List<Role> existingRoles = roleService.readRoleAssignments();
        existingRoles.add(role);
        roleService.writeRoleAssignments(existingRoles);
    }

    public static void clearPermissions(PermissionEditor permissionEditor, RoleService roleService) {
        Map<Role, Set<Permission>> rolePermission = new HashMap<>();
        permissionEditor.editPermissions(GLOBAL_SECURITY_ALIAS, rolePermission);
        roleService.writeRoleAssignments(emptyList());
    }
}
