package com.xebialabs.xlrelease.builder;

import java.util.*;
import java.util.stream.Collectors;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.xlrelease.api.internal.InternalMetadata;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.status.FlagStatus;
import com.xebialabs.xlrelease.domain.status.ReleaseStatus;
import com.xebialabs.xlrelease.domain.variables.FolderVariables;
import com.xebialabs.xlrelease.domain.variables.GlobalVariables;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.repository.Ids;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.builder.TeamBuilder.newTeam;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.PLANNED;
import static com.xebialabs.xlrelease.repository.Ids.formatWithFolderId;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_RELEASE;
import static com.xebialabs.xlrelease.utils.CiHelper.rewriteWithNewId;
import static com.xebialabs.xlrelease.variable.VariableFactory.createVariableByValueType;

public class ReleaseBuilder {

    private ReleaseKind kind = ReleaseKind.RELEASE;
    private String id;
    private String parentReleaseId;
    private Integer ciUid;
    private String title = "Test Title"; //title cannot be null when storing in sql
    private String description;
    private ReleaseStatus status = PLANNED;
    private List<Phase> phases;
    private Date scheduledStartDate;
    private boolean autoStart;
    private Date dueDate;
    private Date startDate;
    private Date endDate;
    private List<Team> teams = Lists.newArrayList();
    private String owner;
    private List<Variable> variables = Lists.newArrayList();
    private List<String> tags = newArrayList();
    private FlagStatus flagStatus = FlagStatus.OK;
    private String flagComment;
    private String calendarToken;
    private boolean calendarPublished;
    private boolean tutorial;
    private boolean abortOnFailure;
    private boolean archiveRelease = true;
    private boolean allowPasswordsInAllFields;
    private boolean disableNotifications;
    private boolean allowConcurrentReleasesFromTrigger = true;
    private List<Attachment> attachments = newArrayList();
    private String originTemplateId;
    private Integer plannedDuration;
    private String scriptUsername;
    private String scriptUserPassword;
    private boolean overdueNotified;
    private String author;
    private TemplateLogo logo;
    private GlobalVariables globalVariables;
    private FolderVariables folderVariables;
    private List<ReleaseExtension> extensions = newArrayList();
    private int maxConcurrency = -1;
    private Map<String, InternalMetadata> metadata = new HashMap<>();
    private Map<String, String> variableMapping = new HashMap<>();
    private CiAttributes ci$attributes = new CiAttributes(null, null, null, null, null);

    private Set<String> categories = new HashSet<>();
    private String defaultTargetFolderId;
    private boolean allowTargetFolderOverride = true;
    private boolean allowRestartInExecutionView = true;

    private String folderId;


    private ReleaseBuilder() {
    }

    public static ReleaseBuilder newRelease() {
        return new ReleaseBuilder();
    }

    public static ReleaseBuilder newTemplate() {
        return new ReleaseBuilder().withStatus(ReleaseStatus.TEMPLATE);
    }

    public ReleaseBuilder withKind(ReleaseKind kind) {
        this.kind = kind;
        return this;
    }

    public ReleaseBuilder withCategories(Set<String> categories) {
        this.categories = categories;
        return this;
    }

    public ReleaseBuilder withId(String id) {
        this.id = id;
        return this;
    }

    public ReleaseBuilder withCiUid(Integer ciUid) {
        this.ciUid = ciUid;
        return this;
    }

    public ReleaseBuilder withTitle(String title) {
        this.title = title;
        return this;
    }

    public ReleaseBuilder withIdAndTitle(String id) {
        this.id = id;
        this.title = id;
        return this;
    }

    public ReleaseBuilder withStatus(ReleaseStatus status) {
        this.status = status;
        return this;
    }

    public ReleaseBuilder completed() {
        this.status = ReleaseStatus.COMPLETED;
        if (this.startDate == null) {
            this.startDate = new Date();
        }
        if (this.endDate == null) {
            this.endDate = new Date();
        }
        return this;
    }

    public ReleaseBuilder withPhases(Phase... phases) {
        this.phases = newArrayList(phases);
        return this;
    }

    public ReleaseBuilder withDescription(String description) {
        this.description = description;
        return this;
    }

    public ReleaseBuilder withScheduledStartDate(Date scheduledStartDate) {
        this.scheduledStartDate = scheduledStartDate;
        return this;
    }

    public ReleaseBuilder withAutoStart(boolean autoStart) {
        this.autoStart = autoStart;
        return this;
    }

    public ReleaseBuilder withDueDate(Date dueDate) {
        this.dueDate = dueDate;
        return this;
    }

    public ReleaseBuilder withStartDate(Date startDate) {
        this.startDate = startDate;
        return this;
    }

    public ReleaseBuilder withEndDate(Date endDate) {
        this.endDate = endDate;
        return this;
    }

    public ReleaseBuilder withCurrentDates() {
        withScheduledStartDate(new Date());
        withDueDate(new Date());
        return this;
    }

    public ReleaseBuilder withPlannedDuration(Integer plannedDuration) {
        this.plannedDuration = plannedDuration;
        return this;
    }

    public ReleaseBuilder withTeams(Team... teams) {
        return withTeams(newArrayList(teams));
    }

    public ReleaseBuilder withTeams(List<Team> teams) {
        List<Team> newTeams = new ArrayList<>(teams);
        newTeams.addAll(this.teams);
        this.teams = newTeams;
        return this;
    }

    public ReleaseBuilder withOwner(String owner) {
        this.owner = owner;
        return this;
    }

    public ReleaseBuilder withOverdueNotified(boolean overdueNotified) {
        this.overdueNotified = overdueNotified;
        return this;
    }

    public ReleaseBuilder withVariableValues(Map<String, ?> variableValues) {
        this.variables.addAll(toVariables(variableValues, false));
        return this;
    }

    public ReleaseBuilder withPasswordVariableValues(Map<String, ?> passwordVariables) {
        this.variables.addAll(toVariables(passwordVariables, true));
        return this;
    }

    public ReleaseBuilder withVariables(List<Variable> variables) {
        this.variables = variables;
        return this;
    }

    public ReleaseBuilder withVariables(Variable... variables) {
        return withVariables(newArrayList(variables));
    }

    public ReleaseBuilder withGlobalVariables(GlobalVariables globalVariables) {
        this.globalVariables = globalVariables;
        return this;
    }

    public ReleaseBuilder withFolderVariables(FolderVariables folderVariables) {
        this.folderVariables = folderVariables;
        return this;
    }

    public ReleaseBuilder withTags(List<String> tags) {
        this.tags = tags;
        return this;
    }

    public ReleaseBuilder withTags(String... tags) {
        return withTags(newArrayList(tags));
    }

    public ReleaseBuilder withFlagStatus(FlagStatus flagStatus) {
        this.flagStatus = flagStatus;
        return this;
    }

    public ReleaseBuilder withFlagComment(String flagComment) {
        this.flagComment = flagComment;
        return this;
    }

    public ReleaseBuilder withCalendarToken(String calendarToken) {
        this.calendarToken = calendarToken;
        return this;
    }

    public ReleaseBuilder withCalendarPublished(boolean calendarPublished) {
        this.calendarPublished = calendarPublished;
        return this;
    }

    public ReleaseBuilder withTutorial(boolean tutorial) {
        this.tutorial = tutorial;
        return this;
    }

    public ReleaseBuilder withAbortOnFailure(boolean abortOnFailure) {
        this.abortOnFailure = abortOnFailure;
        return this;
    }


    public ReleaseBuilder withArchiveRelease(boolean archiveRelease) {
        this.archiveRelease = archiveRelease;
        return this;
    }

    public ReleaseBuilder withAllowPasswordsInAllFields(boolean allowPasswordsInAllFields) {
        this.allowPasswordsInAllFields = allowPasswordsInAllFields;
        return this;
    }

    public ReleaseBuilder withDisableNotifications(boolean disableNotifications) {
        this.disableNotifications = disableNotifications;
        return this;
    }

    public ReleaseBuilder withAllowConcurrentReleasesFromTrigger(boolean val) {
        this.allowConcurrentReleasesFromTrigger = val;
        return this;
    }

    public ReleaseBuilder withAttachments(Attachment... attachments) {
        this.attachments = newArrayList(attachments);
        return this;
    }

    public ReleaseBuilder withOriginTemplateId(String originTemplateId) {
        this.originTemplateId = originTemplateId;
        return this;
    }

    public ReleaseBuilder withMemberViewers(String... members) {
        final Team viewersTeam = createOrFindViewersTeam();
        viewersTeam.addMembers(members);
        return this;
    }

    public ReleaseBuilder withRoleViewers(String... roles) {
        final Team viewersTeam = createOrFindViewersTeam();
        viewersTeam.addRoles(roles);
        return this;
    }

    private Team createOrFindViewersTeam() {
        return this.teams.stream()
                .filter(team -> team.getTeamName().equals("Viewers team"))
                .findFirst().orElseGet(() -> {
                    final Team viewersTeam = newTeam()
                            .withTeamName("Viewers team")
                            .withPermissions(VIEW_RELEASE.getPermissionName())
                            .build();
                    teams.add(viewersTeam);
                    return viewersTeam;
                });
    }

    public ReleaseBuilder withScriptUsername(String scriptUsername) {
        this.scriptUsername = scriptUsername;
        return this;
    }

    public ReleaseBuilder withScriptUserPassword(String scriptUserPassword) {
        this.scriptUserPassword = scriptUserPassword;
        return this;
    }

    public ReleaseBuilder addExtension(ReleaseExtension extension) {
        this.extensions.add(extension);
        return this;
    }

    public ReleaseBuilder withMaxConcurrency(int maxConcurrency) {
        this.maxConcurrency = maxConcurrency;
        return this;
    }

    public ReleaseBuilder withMetadata(String key, InternalMetadata metadata) {
        this.metadata.put(key, metadata);
        return this;
    }

    public ReleaseBuilder withVariableMapping(Map<String, String> variableMapping) {
        this.variableMapping = variableMapping;
        return this;
    }

    public ReleaseBuilder withCiAttributes(CiAttributes ci$Attributes) {
        this.ci$attributes = ci$Attributes;
        return this;
    }

    public ReleaseBuilder withLogo(TemplateLogo logo) {
        this.logo = logo;
        return this;
    }

    public ReleaseBuilder withAuthor(String author) {
        this.author = author;
        return this;
    }

    public ReleaseBuilder withDefaultTargetFolderId(String folderId) {
        this.defaultTargetFolderId = folderId;
        return this;
    }

    public ReleaseBuilder withAllowTargetFolderOverride(boolean allow) {
        this.allowTargetFolderOverride = allow;
        return this;
    }

    public ReleaseBuilder withAllowRestartInExecutionView(boolean allowRestartInExecutionView) {
        this.allowRestartInExecutionView = allowRestartInExecutionView;
        return this;
    }

    public ReleaseBuilder withFolderId(String folderId) {
        this.folderId = folderId;
        return this;
    }

    public ReleaseBuilder withParentReleaseId(String parentReleaseId) {
        this.parentReleaseId = parentReleaseId;
        return this;
    }

    public Release build() {
        final Release release = new Release().getType().getDescriptor().newInstance(id);
        release.setKind(this.kind);
        release.setCiUid(this.ciUid);
        release.setStatus(this.status);
        release.setTitle(this.title);
        release.setDescription(this.description);
        release.setScheduledStartDate(this.scheduledStartDate);
        release.setAutoStart(this.autoStart);
        release.setDueDate(this.dueDate);
        release.setStartDate(this.startDate);
        release.setEndDate(this.endDate);
        release.setPlannedDuration(plannedDuration);
        release.setTags(this.tags);
        release.setFlagStatus(flagStatus);
        release.setFlagComment(flagComment);
        release.setCalendarLinkToken(calendarToken);
        release.setCalendarPublished(calendarPublished);
        release.setTutorial(tutorial);
        release.setAbortOnFailure(abortOnFailure);
        release.setArchiveRelease(archiveRelease);
        release.setAllowPasswordsInAllFields(allowPasswordsInAllFields);
        release.setDisableNotifications(disableNotifications);
        release.setAllowConcurrentReleasesFromTrigger(allowConcurrentReleasesFromTrigger);
        release.setOriginTemplateId(originTemplateId);
        release.setScriptUsername(scriptUsername);
        release.setScriptUserPassword(scriptUserPassword);
        release.setOverdueNotified(overdueNotified);
        release.setExtensions(extensions);
        release.setVariableMapping(variableMapping);
        release.setAuthor(author);
        release.setLogo(logo);
        release.get$metadata().putAll(metadata);
        release.set$ciAttributes(ci$attributes);
        release.setCategories(categories);
        release.setAllowTargetFolderOverride(allowTargetFolderOverride);
        release.setAllowRestartInExecutionView(allowRestartInExecutionView);
        release.setDefaultTargetFolderId(defaultTargetFolderId);
        release.setParentReleaseId(parentReleaseId);

        if (null != this.teams) {
            // Fixup viewers team ID
            this.teams.stream()
                    .filter(team -> "TeamViewers".equals(team.getId()))
                    .forEach(team -> team.setId(id + "/TeamViewers"));
            release.setTeams(this.teams);
        }
        if (null != this.phases) {
            for (Phase phase : phases) {
                phase.setRelease(release);
            }
            release.setPhases(this.phases);
        }
        release.setOwner(this.owner);
        release.setAttachments(attachments);

        if (maxConcurrency >= 0) {
            release.setMaxConcurrentReleases(maxConcurrency);
        }

        release.setQueryableStartDate(release.getStartOrScheduledDate());
        release.setQueryableEndDate(release.getEndOrDueDate());

        release.setVariables(variables);
        release.setGlobalVariables(globalVariables);
        release.setFolderVariables(folderVariables);

        // I do not dare to fix logic in release.fixId(folderId), but this is similar thing
        if (folderId != null) {
            var oldId = release.getId();
            if (oldId != null) {
                rewriteWithNewId(release, formatWithFolderId(folderId, Ids.getFolderlessId(oldId)));
            } else {
                throw new IllegalArgumentException("Release id is not present, but folderId is. Please fix it.");
            }
        }

        return release;
    }

    private List<Variable> toVariables(Map<String, ?> values, boolean isPassword) {
        return values.entrySet()
                .stream()
                .map(e -> createVariableByValueType(e.getKey(), e.getValue(), isPassword, false))
                .collect(Collectors.toList());
    }

}
