package com.xebialabs.xlrelease.builder

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.variables.Variable._
import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.variable.VariableHelper.withoutVariableSyntax

import java.util.{List => JList, Map => JMap, Set => JSet}

object VariableBuilder {
  def newVariable(k: String, v: String): StringVariable = newStringVariable(k, v).build()
  def newGlobalVariable(k: String, v: String): StringVariable = newStringVariable(k, v).withRequiresValue(false).withShowOnReleaseStart(false).build()
  def newStringVariable(k: String, v: String) = new VariableBuilder(classOf[StringVariable], k, v)
  def newPasswordStringVariable(k: String, v: String) = new VariableBuilder(classOf[PasswordStringVariable], k, v)
  def newExternalPasswordVariable(k: String, v: String) = new VariableBuilder(classOf[PasswordStringVariable], k, "").withExternalValue(v)
  def newListStringVariable(k: String, v: JList[String]) = new VariableBuilder(classOf[ListStringVariable], k, v)
  def newSetStringVariable(k: String, v: JSet[String]) = new VariableBuilder(classOf[SetStringVariable], k, v)
  def newMapStringStringVariable(k: String, v: JMap[String, String]) = new VariableBuilder(classOf[MapStringStringVariable], k, v)
  def newBooleanVariable(k: String, v: java.lang.Boolean) = new VariableBuilder(classOf[BooleanVariable], k, v)
  def newIntegerVariable(k: String, v: java.lang.Integer) = new VariableBuilder(classOf[IntegerVariable], k, v)
  def newDateVariable(k: String, v: java.util.Date) = new VariableBuilder(classOf[DateVariable], k, v)
}

class VariableBuilder[V <: AnyRef, C <: VariableWithValue[V]]
(variableClass: Class[C], key: String, value: V) {

  private var id: String = _
  private var requiresValue: Boolean = true
  private var showOnReleaseStart: Boolean = true
  private var label: String = _
  private var folderId: String = _
  private var description: String = _
  private var valueProvider: ValueProviderConfiguration = _
  private var externalValue: String = _
  private var inherited: Boolean = false
  private var preventInterpolation: Boolean = false

  def withId(id: String): VariableBuilder[V, C] = {
    this.id = id
    this
  }

  def withRequiresValue(value: Boolean): VariableBuilder[V, C] = {
    this.requiresValue = value
    this
  }

  def withShowOnReleaseStart(value: Boolean): VariableBuilder[V, C] = {
    this.showOnReleaseStart = value
    this
  }

  def withLabel(label: String): VariableBuilder[V, C] = {
    this.label = label
    this
  }

  def withFolderId(folderId: String): VariableBuilder[V, C] = {
    this.folderId = folderId
    this
  }

  def withDescription(description: String): VariableBuilder[V, C] = {
    this.description = description
    this
  }

  def withValueProvider(valueProvider: ValueProviderConfiguration): VariableBuilder[V, C] = {
    this.valueProvider = valueProvider
    this
  }

  def withExternalValue(externalValue: String): VariableBuilder[V, C] = {
    this.externalValue = externalValue
    this
  }

  def withPreventInterpolation(preventInterpolation: Boolean): VariableBuilder[V, C] = {
    this.preventInterpolation = preventInterpolation
    this
  }

  def isInherited(inherited: Boolean): VariableBuilder[V, C] = {
    this.inherited = inherited
    this
  }

  def build(): C = {
    val variable: C = Type.valueOf(variableClass).getDescriptor.newInstance(id)
    variable.setKey(withoutVariableSyntax(key))
    variable.setValue(value)
    variable.setRequiresValue(requiresValue)
    variable.setShowOnReleaseStart(showOnReleaseStart)
    variable.setLabel(label)
    variable.setFolderId(folderId)
    variable.setDescription(description)
    if (valueProvider != null) {
      if (id != null) {
        valueProvider.setId(s"$id/${ValueProviderConfiguration.CONFIGURATION_ID}")
      }
      valueProvider.setVariable(variable)
      variable.setValueProvider(valueProvider)
    }
    variable.setInherited(inherited)
    variable match {
      case stringVar: StringVariable => stringVar.setPreventInterpolation(preventInterpolation)
      case _ => if (this.preventInterpolation)
        throw new IllegalArgumentException(s"Preventing interpolation is not supported on a variable of type $variableClass")
    }
    variable
  }

}
