package com.xebialabs.xlrelease.domain;

import java.util.LinkedHashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.XLRelease;
import com.xebialabs.xlrelease.api.internal.InternalMetadata;

import static com.xebialabs.xlrelease.repository.Ids.isNullId;
import static com.xebialabs.xlrelease.repository.Ids.isPhaseId;
import static com.xebialabs.xlrelease.repository.Ids.isReleaseId;
import static com.xebialabs.xlrelease.repository.Ids.isTaskId;
import static com.xebialabs.xlrelease.variable.VariableHelper.containsVariables;
import static java.lang.String.format;

/**
 * Dependency is a link from a gate task to a plan item (release, phase or task) on which the gate depends.
 * <p>
 * There are several possible states of a dependency:
 * <ul>
 * <li>
 * Variable dependency. In this case the {@link #targetId} contains a variable expression like <i>${varPhaseId}</i>,
 * and {@link #target} is null.
 * </li>
 * <li>
 * Dependency with string target. In this case the {@link #targetId} contains a plan item ID, and {@link #target} is null.
 * <ul>
 * <li>
 * It is possible that {@link #targetId} was emptied by a public API call or by deleting a variable. Then such a
 * dependency does not contain neither {@link #target} nor {@link #targetId} and will cause a Gate to fail.
 * </li>
 * </ul>
 * </li>
 * <li>
 * Normal dependency. In this case the {@link #target} field points to the plan item on which the {@link #gateTask} depends.
 * </li>
 * <li>
 * Archived dependency. A dependency becomes "archived" when the target plan item was deleted, for example when
 * archiving old completed releases. In this case the {@link #target} is {@code null}, and the information about
 * archived plan item is stored in fields {@link #archivedTargetId}, {@link #archivedTargetTitle} and {@link #archivedAsResolved}.
 * The latter shows if the plan item was {@code COMPLETED} or {@code ABORTED}.
 * </li>
 * </ul>
 */
@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(description = "A dependency to another release, phase or task in a Gate task.", versioned = false)
public class Dependency extends BaseConfigurationItem implements CiWithInternalMetadata {
    private static final Logger logger = LoggerFactory.getLogger(Dependency.class);

    public static final String TYPE_DEPENDENCY = XLRelease.PREFIX + "." + Dependency.class.getSimpleName();
    public static final String PROPERTY_TARGET = "target";

    @Property(asContainment = true, description = "The source gate task from the dependency.")
    private GateTask gateTask;

    @Property(required = false, description = "The target of the dependency (release, phase, or task).")
    private PlanItem target;

    @Property(required = false, description = "The target ID of the dependency (when it is filled, the `target` will be empty).")
    private String targetId;

    @Property(required = false, description = "The target title once it is archived.")
    private String archivedTargetTitle;

    @Property(required = false, description = "The target ID once it is archived.")
    private String archivedTargetId;

    @Property(required = false, description = "Shows if the plan item was `COMPLETED` or `ABORTED`.")
    private boolean archivedAsResolved;

    /**
     * An additional derived Dependency metadata.
     * Does not belong to public API and can be changed.
     */
    private Map<String, InternalMetadata> $metadata = new LinkedHashMap<>();


    @PublicApiMember
    public GateTask getGateTask() {
        return gateTask;
    }

    @PublicApiMember
    public void setGateTask(GateTask gateTask) {
        this.gateTask = gateTask;
    }

    @SuppressWarnings("unchecked")
    public <T extends PlanItem> T getTarget() {
        return (T) target;
    }

    public void setTarget(PlanItem target) {
        if (null != target && null != target.getId()) {
            this.targetId = target.getId();
        }
        this.target = target;
    }

    public String getTargetId() {
        if (null == targetId && null != target) {
            targetId = target.getId();
        }
        return targetId;
    }

    public void setTargetId(final String targetId) {
        this.targetId = targetId;
    }

    @PublicApiMember
    public String getArchivedTargetId() {
        return archivedTargetId;
    }

    @PublicApiMember
    public String getArchivedTargetTitle() {
        return archivedTargetTitle;
    }

    @PublicApiMember
    public boolean isArchived() {
        return archivedTargetId != null || archivedTargetTitle != null;
    }

    @PublicApiMember
    public boolean isDone() {
        if (isArchived()) {
            logger.trace("isDone isArchived() = true, returning {}", archivedAsResolved);
            return archivedAsResolved;
        }
        //noinspection SimplifiableIfStatement
        if (!hasResolvedTarget()) {
            logger.trace("isDone returns false because hasResolvedTarget() = false, targetId = {}, archivedTargetId = {}, isArchived = {}", targetId, archivedTargetId, isArchived());
            return false;
        }
        logger.trace("isDone returns getTarget().isDone = {}, {}", getTarget().isDone(), getTarget() instanceof Release ? ((Release) getTarget()).getStatus() : null);
        return getTarget().isDone();
    }

    @PublicApiMember
    public boolean isAborted() {
        if (isArchived()) {
            return !archivedAsResolved;
        }
        //noinspection SimplifiableIfStatement
        if (!hasResolvedTarget()) {
            return false;
        }
        return getTarget().isAborted();
    }

    @PublicApiMember
    public String getTargetDisplayPath() {
        if (isArchived()) {
            return getArchivedTargetTitle();
        }
        if (hasResolvedTarget()) {
            return getTarget().getDisplayPath();
        }
        return getTargetId();
    }

    @PublicApiMember
    public String getTargetTitle() {
        if (isArchived()) {
            return getArchivedTargetTitle();
        }
        if (hasResolvedTarget()) {
            return getTarget().getTitle();
        }
        return getTargetId();
    }

    public void archive() {
        if (hasResolvedTarget()) {
            this.archivedTargetId = target.getId();
            this.archivedAsResolved = target.isDone();
            this.archivedTargetTitle = createArchivedTargetTitle(target);
            this.target = null;
        } else {
            this.archivedTargetId = targetId;
            this.archivedAsResolved = false;
            this.archivedTargetTitle = targetId;
            this.targetId = null;
        }
    }

    public boolean isArchivedAsResolved() {
        return archivedAsResolved;
    }

    public static String createArchivedTargetTitle(PlanItem planItem) {
        // ENG-8209 archiving of a completed release will trigger proxy initialization of another release
        //  because we need fully loaded target proxy object to get "archivedTargetTitle"
        if (planItem instanceof Release) {
            return planItem.getTitle();
        } else if (planItem instanceof Phase) {
            return format("%s / %s", planItem.getRelease().getTitle(), planItem.getTitle());
        } else if (planItem instanceof Task) {
            return format("%s / %s / %s", planItem.getRelease().getTitle(),
                ((Task) planItem).getPhase().getTitle(), planItem.getTitle());
        }
        return null;
    }

    public boolean hasResolvedTarget() {
        return target != null;
    }

    public boolean hasVariableTarget() {
        return containsVariables(targetId);
    }

    public boolean hasValidArchiveTargetId() {
        return !isNullId(archivedTargetId) && isValidAllowedPlanItemId();
    }

    private boolean isValidAllowedPlanItemId() {
        return isReleaseId(archivedTargetId) || isPhaseId(archivedTargetId) || isTaskId(archivedTargetId);
    }

    @Override
    public Map<String, InternalMetadata> get$metadata() {
        return $metadata;
    }
}
