package com.xebialabs.xlrelease.domain;

import java.util.List;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;

import static com.google.common.collect.Sets.newHashSet;
import static java.util.stream.Collectors.toList;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(label = "Parallel Group", versioned = false)
public class ParallelGroup extends TaskGroup {

    @Property(asContainment = true, required = false)
    protected Set<Link> links = newHashSet();

    @PublicApiMember
    @Override
    public void setTasks(List<Task> tasks) {
        this.tasks = tasks;
    }

    @PublicApiMember
    public Set<Link> getLinks() {
        return links;
    }

    @PublicApiMember
    public void setLinks(Set<Link> links) {
        this.links = links;
    }

    public List<Link> getLinksOf(final Task task) {
        return links.stream()
                .filter(link -> link.references(task))
                .collect(toList());
    }


    public List<Task> getPlanningTargetsOf(final Task task) {
        List<Link> targetLinks = links.stream()
                .filter(link -> link.hasSource(task))
                .collect(toList());
        return targetLinks.stream().map(Link::getTarget).collect(toList());
    }

    public List<Task> getPlanningSourceOf(final Task task) {
        List<Link> sourceLinks = links.stream()
                .filter(link -> link.hasTarget(task))
                .collect(toList());
        return sourceLinks.stream().map(Link::getSource).collect(toList());
    }

    @Override
    protected Changes tryToStartPlanningTargets(Task task) {
        Changes changes = new Changes();
        for (Task target : getPlanningTargetsOf(task)) {
            if (!hasActiveSource(target) && !target.isDefunct()) {
                changes.addAll(target.start());
                if (target.isDone()) {
                    changes.addAll(tryToStartPlanningTargets(target));
                }
            }
        }
        return changes;
    }

    private boolean hasActiveSource(Task task) {
        List<Task> sources = getPlanningSourceOf(task);
        return sources.stream().anyMatch(source -> !source.isDefunct());
    }

    @Override
    protected Changes startSubTasksIfPreconditionNotInProgress() {
        Changes changes = new Changes();

        if (!isPreconditionInProgress()) {
            for (Task subTask : tasks) {
                if (getPlanningSourceOf(subTask).isEmpty()) {
                    changes.addAll(subTask.start());
                    if (subTask.isDone()) {
                        changes.addAll(tryToStartPlanningTargets(subTask));
                    }
                }
            }
        }

        return changes;
    }

    public Link findLink(String linkId) {
        return links.stream().filter(link -> link.getId().equals(linkId)).findFirst().orElseGet(null);
    }

}
