package com.xebialabs.xlrelease.domain;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.repository.core.Securable;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.internal.InternalMetadata;
import com.xebialabs.xlrelease.domain.status.FlagStatus;
import com.xebialabs.xlrelease.domain.status.PhaseStatus;
import com.xebialabs.xlrelease.domain.status.ReleaseStatus;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import com.xebialabs.xlrelease.domain.variables.FolderVariables;
import com.xebialabs.xlrelease.domain.variables.GlobalVariables;
import com.xebialabs.xlrelease.domain.variables.StringVariable;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.domain.variables.reference.*;
import com.xebialabs.xlrelease.events.*;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.risk.domain.RiskProfile;
import com.xebialabs.xlrelease.user.User;
import com.xebialabs.xlrelease.variable.VariableHelper;
import com.xebialabs.xlrelease.variable.ValueWithInterpolation;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS;
import static com.xebialabs.xlrelease.builder.VariableBuilder.newStringVariable;
import static com.xebialabs.xlrelease.domain.Task.IS_AUTOMATED_AND_IN_PROGRESS;
import static com.xebialabs.xlrelease.domain.status.FlagStatus.OK;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.*;
import static com.xebialabs.xlrelease.domain.variables.FolderVariables.FOLDER_VARIABLES;
import static com.xebialabs.xlrelease.domain.variables.GlobalVariables.GLOBAL_VARIABLES;
import static com.xebialabs.xlrelease.repository.Ids.formatWithFolderId;
import static com.xebialabs.xlrelease.repository.Ids.isInRootFolder;
import static com.xebialabs.xlrelease.utils.CiHelper.removeCisWithId;
import static com.xebialabs.xlrelease.utils.CiHelper.rewriteWithNewId;
import static com.xebialabs.xlrelease.variable.VariableFactory.createVariableByValueType;
import static com.xebialabs.xlrelease.variable.VariableFactory.variableReferencesToVariables;
import static com.xebialabs.xlrelease.variable.VariableHelper.*;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

@Metadata(description = "A release or template.", root = APPLICATIONS, versioned = false)
@PublicApiRef
@ShowOnlyPublicApiMembers
public class Release extends PlanItem implements Securable, CiWithInternalMetadata, VariableContainer, CiWithUid {
    public static String SCRIPT_USER_PASSWORD_VARIABLE_MAPPING_KEY = "scriptUserPassword";

    @Property(category = "internal", required = false)
    private String rootReleaseId;

    @Property(category = "internal", defaultValue = "100", description = "The maximum number of concurrent releases that can be started by Create Release tasks")
    private int maxConcurrentReleases;

    @Deprecated
    @Property(asContainment = true, required = false, description = "The triggers that may start a release from a template. (Templates only)", isTransient = true)
    protected List<ReleaseTrigger> releaseTriggers = new ArrayList<>();

    /**
     * The Digital.ai Release teams are transient CIs and are kept for backwards-compatibility. The teams' data is stored using platform's
     * {@link com.xebialabs.deployit.security.Role}. The teams need to be exported to template JSONs so they have to be a UDM property.
     */
    @Property(asContainment = true, required = false, description = "The teams configured on the release.", isTransient = true)
    protected List<Team> teams = new ArrayList<>();

    /**
     * memberViewers field is kept for backwards compatibility with JSON deserializer but not used anymore.
     */
    @SuppressWarnings("unused")
    @Deprecated
    @Property(required = false, category = "internal", isTransient = true)
    private List<String> memberViewers = new ArrayList<>();

    /**
     * roleViewers field is kept for backwards compatibility with JSON deserializer but not used anymore.
     */
    @SuppressWarnings("unused")
    @Deprecated
    @Property(required = false, category = "internal", isTransient = true)
    private List<String> roleViewers = new ArrayList<>();

    @Property(asContainment = true, required = false, description = "File attachments of the release.")
    private List<Attachment> attachments = new ArrayList<>();

    @Property(asContainment = true, required = false, description = "The list of phases in the release.")
    protected List<Phase> phases = new ArrayList<>();

    @Property(required = false, category = "internal")
    protected Date queryableStartDate;

    @Property(required = false, category = "internal")
    protected Date queryableEndDate;

    @Property(description = "The calculated flag status, derived from the flags from the release and its tasks.", defaultValue = "OK")
    protected FlagStatus realFlagStatus = FlagStatus.OK;

    @Property(description = "The state the release is in.")
    protected ReleaseStatus status;

    @Property(required = false, description = "The tags of the release. Tags can be used for grouping and querying.")
    protected List<String> tags;

    @Property(asContainment = true, required = false, description = "List of variable CIs representing variables in this release or template")
    protected List<Variable> variables = new ArrayList<>();

    @Property(required = false, category = "internal")
    protected String calendarLinkToken;

    @Property(required = false, category = "internal")
    protected boolean calendarPublished;

    @Property(required = false, category = "internal")
    protected boolean tutorial;

    @Property(required = false, description = "Releases automatically abort when a task fails if this property is set to true.")
    protected boolean abortOnFailure;

    @Property(required = false, defaultValue = "true", description = "Archive release")
    protected boolean archiveRelease = true;

    @Property(required = false, description = "Allows the usage of passwords in non-password fields. Passwords values are masked in the UI and logging output but are decrypted before the task execution.")
    protected boolean allowPasswordsInAllFields;

    @Property(required = false, description = "Disable all notifications for this release.")
    protected boolean disableNotifications;

    @Property(required = false, description = "If set to false, a trigger can't create a release if the previous one it created is still running.")
    protected boolean allowConcurrentReleasesFromTrigger = true;

    @Property(required = false, description = "The ID of the template that created this release.")
    protected String originTemplateId;

    @Property(required = false, description = "True if release was created by a trigger.")
    protected boolean createdFromTrigger;

    // Also referred to as the automated tasks user
    @Property(required = false, description = "The credentials of this user are used to run automated scripts in this release.")
    protected String scriptUsername;

    @Property(required = false, password = true, description = "The password of the user that lends his credentials to run the scripts.")
    protected String scriptUserPassword;

    @Property(required = false, asContainment = true, description = "Extensions of this release, e.g. 'Release contents dashboard'")
    protected List<ReleaseExtension> extensions = new ArrayList<>();

    @Property(required = false, description = "The Create Release task from which this release was created, if any")
    protected String startedFromTaskId;

    @Property(required = false, defaultValue = "false", description = "If true, the release will automatically start at scheduledStartDate.")
    protected boolean autoStart;

    @Property(category = "internal")
    protected int automatedResumeCount = 0;

    @Property(category = "internal", hidden = true, defaultValue = "50", description = "The maximum number of automatic release resumes performed during phase restarts.")
    protected int maxAutomatedResumes;

    @Property(required = false, description = "The comment to associate with the action")
    private String abortComment;

    @Property(asContainment = true, required = false, description = "Map from property name to a variable name that replaces that property")
    private Map<String, String> variableMapping = new HashMap<>();

    @Property(required = false, description = "Risk profile used in risk calculations")
    protected RiskProfile riskProfile;

    /**
     * An additional derived Release metadata that is used in our UI. Does not
     * belong to public API and can be changed. See implementations of com.xebialabs.xlrelease.api.internal.InternalMetadataDecorator
     * for usages.
     */
    private Map<String, InternalMetadata> $metadata = new LinkedHashMap<>();

    private boolean archived;

    private Integer ciUid;

    private static final Logger logger = LoggerFactory.getLogger(Release.class);

    @PublicApiMember
    public List<Phase> getPhases() {
        return phases;
    }

    public void setPhases(List<Phase> phases) {
        this.phases = phases;
    }

    @Deprecated
    @PublicApiMember
    public List<ReleaseTrigger> getReleaseTriggers() {
        return releaseTriggers;
    }

    @Deprecated
    public Optional<ReleaseTrigger> getReleaseTriggerById(String releaseTriggerId) {
        return releaseTriggers.stream().filter(releaseTrigger -> releaseTrigger.getId().equals(releaseTriggerId)).findFirst();
    }

    @Deprecated
    public void setReleaseTriggers(List<ReleaseTrigger> releaseTriggers) {
        this.releaseTriggers = releaseTriggers;
    }

    @Deprecated
    public void deleteReleaseTriggerById(String triggerId) {
        releaseTriggers = releaseTriggers.stream().filter(releaseTrigger -> !releaseTrigger.getId().equals(triggerId)).collect(toList());
    }

    @Deprecated
    public void replaceTrigger(ReleaseTrigger releaseTrigger) {
        Optional<ReleaseTrigger> maybeTrigger = getReleaseTriggerById(releaseTrigger.getId());
        if (maybeTrigger.isPresent()) {
            int index = releaseTriggers.indexOf(maybeTrigger.get());
            releaseTriggers.set(index, releaseTrigger);
        }
    }

    public String getRootReleaseId() {
        return rootReleaseId;
    }

    public void setRootReleaseId(String rootReleaseId) {
        this.rootReleaseId = rootReleaseId;
    }

    public int getMaxConcurrentReleases() {
        return maxConcurrentReleases;
    }

    public void setMaxConcurrentReleases(int maxConcurrentReleases) {
        this.maxConcurrentReleases = maxConcurrentReleases;
    }

    public Integer getCiUid() {
        return ciUid;
    }

    public void setCiUid(final Integer ciUid) {
        this.ciUid = ciUid;
    }

    public int getAutomatedResumeCount() {
        return automatedResumeCount;
    }

    public void setAutomatedResumeCount(final int automatedResumeCount) {
        this.automatedResumeCount = automatedResumeCount;
    }

    public int getMaxAutomatedResumes() {
        return maxAutomatedResumes;
    }

    public void setMaxAutomatedResumes(final int maxAutomatedResumes) {
        this.maxAutomatedResumes = maxAutomatedResumes;
    }

    public String getAbortComment() {
        return abortComment;
    }

    private void setAbortComment(String abortComment) {
        this.abortComment = abortComment;
    }

    /**
     * This method is kept for backwards compatibility. {@link #getVariables()} method provides richer
     * access to the variable management.
     *
     * @return mapping from variable name to variable value containing only variables with string values.
     * Variable names are in <code>${key}</code> format.
     */
    @PublicApiMember
    public Map<String, String> getVariableValues() {
        return VariableHelper.getVariableValuesAsStrings(variables);
    }

    /**
     * This method is kept for backwards compatibility. {@link #getVariables()} method provides richer
     * access to the variable management.
     *
     * @return mapping from variable name to variable value containing only variables with single password values.
     * Variable names are in <code>${key}</code> format.
     */
    @PublicApiMember
    public Map<String, String> getPasswordVariableValues() {
        Map<String, String> values = new HashMap<>();
        values.putAll(VariableHelper.getPasswordVariableValuesAsStrings(variables));
        if (getGlobalVariables() != null) {
            values.putAll(getGlobalVariables().getPasswordVariableValues());
        }
        if (getFolderVariables() != null) {
            values.putAll(getFolderVariables().getPasswordVariableValues());
        }
        return values;
    }

    public List<Variable> getCiPropertyVariables() {
        return Arrays.stream(ReleasePropertyVariableKey.values())
                .filter(property -> property.getValue(this) != null)
                .map(property -> newStringVariable(property.getKey(), property.getValue(this)).build())
                .collect(toList());
    }

    public Map<String, String> getAllStringVariableValues() {
        List<Variable> allVariables = getAllReleaseGlobalAndFolderVariables();
        return VariableHelper.getVariableValuesAsStrings(allVariables);
    }

    private List<Variable> getAllReleaseGlobalAndFolderVariables() {
        List<Variable> allVariables = new ArrayList<>(variables);

        if (getGlobalVariables() != null) {
            allVariables.addAll(this.getGlobalVariables().getVariables());
        }
        if (getFolderVariables() != null) {
            allVariables.addAll(getFolderVariables().getVariables());
        }
        allVariables.addAll(getCiPropertyVariables());

        return allVariables;
    }

    public Map<String, ValueWithInterpolation> getAllVariableValuesAsStringsWithInterpolationInfo() {
        List<Variable> allVars = getAllReleaseGlobalAndFolderVariables();

        Map<String, String> allResolvedVarsAsStrings = VariableHelper.getVariableValuesAsStrings(allVars);
        Map<String, Variable> allVarsMap = allVars.stream().collect(Collectors.toMap(key -> withVariableSyntax(key.getKey()), Function.identity()));

        Map<String, ValueWithInterpolation> varsWithInterpolation = new HashMap<>();
        for (Map.Entry<String, String> entry: allResolvedVarsAsStrings.entrySet()) {
            String value = entry.getValue();
            String key = entry.getKey();

            Variable variable = allVarsMap.get(key);
            boolean preventInterpolation = variable instanceof StringVariable && ((StringVariable)variable).isPreventInterpolation();

            varsWithInterpolation.put(key, new ValueWithInterpolation(value, preventInterpolation));
        }
        return varsWithInterpolation;
    }

    public Set<String> getVariablesKeysInNonInterpolatableVariableValues() {
        return getAllReleaseGlobalAndFolderVariables().stream()
                .filter(v -> v instanceof StringVariable && ((StringVariable) v).isPreventInterpolation())
                .map(v -> VariableHelper.collectVariables(v.getValueAsString()))
                .flatMap(value -> value.stream().map(v -> VariableHelper.withoutVariableSyntax(v)))
                .collect(Collectors.toSet());
    }

    public Map<String, Variable> getVariablesByKeys() {
        return indexByKey(variables);
    }

    /**
     * This method is kept for backwards compatibility. {@link #setVariables(List)} method provides richer
     * access to the variable management.
     *
     * @param variableValues mapping from variable name to variable value containing only variables with string values.
     *                       Variable names are in <code>${key}</code> or <code>key</code> format.
     */
    @PublicApiMember
    public void setVariableValues(Map<String, ?> variableValues) {
        setVariableValues(variableValues, false);
    }

    /**
     * This method is kept for backwards compatibility. {@link #setVariables(List)} method provides richer
     * access to the variable management.
     *
     * @param variableValues mapping from password variable name to password variable value.
     *                       Variable names are in <code>${key}</code> or <code>key</code> format.
     */
    @PublicApiMember
    public void setPasswordVariableValues(Map<String, ?> variableValues) {
        setVariableValues(variableValues, true);
    }

    private void setVariableValues(Map<String, ?> values, boolean passwords) {
        scanAndAddNewVariables();
        for (Map.Entry<String, ?> entry : values.entrySet()) {
            Map<String, Variable> variablesMap = getVariablesByKeys();
            String key = withoutVariableSyntax(entry.getKey());
            if (variablesMap.containsKey(key) || variablesMap.containsKey(entry.getKey())) {
                Variable variable = variablesMap.containsKey(key) ? variablesMap.get(key) : variablesMap.get(entry.getKey());

                if (variable.isPassword() == passwords) {
                    variable.setUntypedValue(entry.getValue());
                }
            } else {
                addVariable(createVariableByValueType(key, entry.getValue(), passwords, false));
            }
        }
    }

    public List<Variable> scanAndAddNewVariables(VisitableItem updated) {
        Set<VariableReference> variableUsages = collectVariableReferences(updated);
        final Map<String, Variable> currentVariables = getVariablesByKeys();
        final Set<String> variableKeysInNonInterpolatableValues = getVariablesKeysInNonInterpolatableVariableValues();

        List<VariableReference> newVariableReferences = variableUsages.stream()
                .filter(v -> !withoutVariableSyntax(v.getKey()).startsWith("folder."))
                .filter(v -> !withoutVariableSyntax(v.getKey()).startsWith("global."))
                .filter(v -> !currentVariables.containsKey(withoutVariableSyntax(v.getKey())))
                .filter(v -> !variableKeysInNonInterpolatableValues.contains(withoutVariableSyntax(v.getKey())))
                .collect(toList());

        if (this.variables == null) {
            this.variables = new ArrayList<>();
        }

        List<Variable> variables = variableReferencesToVariables(newVariableReferences);
        this.variables.addAll(variables);

        return variables;

    }

    public List<Variable> scanAndAddNewVariables() {
        return scanAndAddNewVariables(this);
    }

    public List<Variable> getAllVariables() {
        ArrayList<Variable> allVariables = new ArrayList<>(variables);
        allVariables.addAll(getCiPropertyVariables());
        return allVariables;
    }

    @PublicApiMember
    public List<Variable> getVariables() {
        return Collections.unmodifiableList(variables);
    }

    public void addVariables(List<Variable> newVariables) {
        variables.addAll(newVariables);
    }

    public void replaceVariable(Variable current, Variable replacement) {
        int index = variables.indexOf(current);
        variables.set(index, replacement);
    }

    @PublicApiMember
    public void setVariables(List<Variable> variables) {
        checkVariables(variables);
        this.variables = new ArrayList<>(variables);
    }

    public Optional<Variable> getVariableById(String variableId) {
        return getVariables().stream().filter(variable -> variable.getId().equals(variableId)).findFirst();
    }

    public Variable addVariable(Variable variable) {
        checkVariableCanBeAdded(variable);
        this.variables.add(variable);
        return variable;
    }

    public void checkVariableCanBeAdded(Variable variable) {
        checkVariable(variable);
        checkArgument(!getVariablesByKeys().containsKey(variable.getKey()), "A variable already exists by key '%s'", variable.getKey());
    }

    public GlobalVariables getGlobalVariables() {
        return (GlobalVariables) $metadata.get(GLOBAL_VARIABLES);
    }

    public void setGlobalVariables(GlobalVariables globalVariables) {
        this.$metadata.put(GLOBAL_VARIABLES, globalVariables);
    }

    public FolderVariables getFolderVariables() {
        return (FolderVariables) $metadata.get(FOLDER_VARIABLES);
    }

    public void setFolderVariables(FolderVariables folderVariables) {
        this.$metadata.put(FOLDER_VARIABLES, folderVariables);
    }

    public Changes removeVariable(final String variableId) {
        Changes changes = new Changes();

        removeCisWithId(this.variables, variableId);
        changes.remove(variableId);
        changes.update(this);

        List<UserInputTask> userInputTasks = getAllTasksOfType(UserInputTask.class);
        for (UserInputTask task : userInputTasks) {
            changes.addAll(task.removeVariable(variableId));
        }

        return changes;
    }

    public boolean isVariableUsed(Variable variable) {
        final String variableKey = withVariableSyntax(variable.getKey());
        final Set<VariableReference> variableReferences = collectVariableReferences();
        return variableReferences.stream().anyMatch(input -> variableKey.equals(input.getKey()));
    }

    @PublicApiMember
    public ReleaseStatus getStatus() {
        return status;
    }

    @PublicApiMember
    public void setStatus(ReleaseStatus value) {
        this.status = value;
    }

    public Team getAdminTeam() {
        for (Team team : getTeams()) {
            if (team.getTeamName().equals(Team.RELEASE_ADMIN_TEAMNAME)) {
                return team;
            }
        }
        return null;
    }

    @PublicApiMember
    public List<Team> getTeams() {
        return teams;
    }

    public void setTeams(List<Team> teams) {
        this.teams = teams;
    }

    @PublicApiMember
    public List<String> getTags() {
        return tags;
    }

    @PublicApiMember
    public void setTags(List<String> tags) {
        this.tags = tags != null ? tags.stream().distinct().collect(Collectors.toList()) : Collections.emptyList();
    }

    @PublicApiMember
    public List<Attachment> getAttachments() {
        return attachments;
    }

    public void setAttachments(List<Attachment> attachments) {
        this.attachments = attachments;
    }

    public void deleteAttachment(String attachmentId) {
        removeCisWithId(attachments, attachmentId);
    }

    public Set<Task> getTasksUsingAttachment(String attachmentId) {
        Set<Task> tasks = new HashSet<>();

        for (Task task : getAllTasks()) {
            tasks.addAll(task.getAttachments().stream()
                    .filter(attachment -> attachment.getId().equals(attachmentId))
                    .map(attachment -> task)
                    .collect(Collectors.toSet()));
        }

        return tasks;
    }


    public boolean hasBeenStarted() {
        return status != null && status.hasBeenStarted();
    }

    public List<Attachment> getReleaseAttachments() {
        Set<Attachment> attachments = new HashSet<>(this.attachments);

        List<Task> allTasks = getAllTasks();
        for (Task task : allTasks) {
            attachments.removeAll(task.getAttachments());
        }

        return new ArrayList<>(attachments);
    }

    @PublicApiMember
    public String getCalendarLinkToken() {
        return calendarLinkToken;
    }

    public void setCalendarLinkToken(String calendarLinkToken) {
        this.calendarLinkToken = calendarLinkToken;
    }

    @PublicApiMember
    public boolean isCalendarPublished() {
        return calendarPublished;
    }

    @PublicApiMember
    public void setCalendarPublished(boolean calendarPublished) {
        this.calendarPublished = calendarPublished;
    }

    public List<ReleaseExtension> getExtensions() {
        return extensions;
    }

    public void setExtensions(List<ReleaseExtension> extensions) {
        this.extensions = extensions;
    }

    public FlagStatus getRealFlagStatus() {
        return realFlagStatus;
    }

    public Phase getCurrentPhase() {
        return getPhases().stream()
                .filter(phase -> phase.getStatus() == PhaseStatus.IN_PROGRESS || phase.isFailing() || phase.isFailed())
                .findFirst().orElse(null);
    }

    public Task getCurrentTask() {
        Phase currentPhase = getCurrentPhase();
        if (currentPhase != null) {
            return currentPhase.getCurrentTask();
        }

        return null;
    }

    public boolean hasCurrentPhase() {
        return null != getCurrentPhase();
    }

    @VisibleForTesting
    void complete(Changes changes) {
        setStatus(COMPLETED);
        setEndDate(new Date());
        setFlagStatus(OK);
        setFlagComment("");
        freezeVariables();

        changes.update(this);
        changes.addOperation(new ReleaseCompleteOperation(this));
    }

    public Changes start() {
        return start(false);
    }

    public Changes start(boolean releaseStartedImmediatelyAfterBeingCreated) {
        return start(false, releaseStartedImmediatelyAfterBeingCreated);
    }

    public Changes startAsPartOfBulkOperation() {
        return start(true, false);
    }

    public Changes resume() {
        Changes changes = new Changes();

        this.setStatus(IN_PROGRESS);
        if (this.getStartDate() == null) {
            this.setStartDate(new Date());
        }
        changes.update(this);
        changes.addOperation(new ReleaseResumeOperation(this));

        Optional<Phase> firstPlannedPhase = getPhases().stream().filter(Phase::isPlanned).findFirst();
        if (firstPlannedPhase.isPresent()) {
            startActivablePhase(firstPlannedPhase.get(), changes);
        } else {
            throw new IllegalStateException("No planned phase found in release " + getId());
        }

        return changes;
    }

    private boolean hasPhase() {
        return !getPhases().isEmpty();
    }

    private void startActivablePhase(Phase phase, Changes changes) {
        changes.addAll(phase.start());
        checkPhaseStatus(phase, changes);
    }

    private void checkPhaseStatus(Phase phase, Changes changes) {
        if (phase.isDone()) {
            startNextPhase(phase, changes);
        } else if (phase.isFailed()) {
            changes.addAll(fail());
        } else if (phase.isFailing()) {
            changes.addAll(failing());
        }
    }

    private void startNextPhase(Phase phase, Changes changes) {
        if (hasNextPhase(phase)) {
            Phase nextPhase = getNextPhase(phase);
            startActivablePhase(nextPhase, changes);
        } else {
            complete(changes);
        }
    }

    public Changes markTaskAsDone(String taskId, TaskStatus status) {
        Changes changes = new Changes();

        if (hasCurrentPhase()) {
            Phase currentPhase = getCurrentPhase();
            recoverReleaseIfNeeded(changes);
            changes.addAll(currentPhase.markTaskAsDone(taskId, status));
            checkPhaseStatus(currentPhase, changes);
        }
        return changes;
    }

    private void recoverReleaseIfNeeded(Changes changes) {
        if (getStatus() == FAILED || getStatus() == FAILING) {
            changes.update(this);
            changes.addOperation(new ReleaseRetryOperation(this));
            setStatus(IN_PROGRESS);
        }
    }

    public Changes startPendingTask(String taskId) {
        Changes changes = new Changes();

        if (hasCurrentPhase()) {
            Phase currentPhase = getCurrentPhase();
            changes.addAll(currentPhase.startPendingTask(taskId));
            checkPhaseStatus(currentPhase, changes);
        }

        return changes;
    }

    public Changes startWithInput(String taskId) {
        Changes changes = new Changes();

        if (hasCurrentPhase()) {
            Phase currentPhase = getCurrentPhase();
            changes.addAll(currentPhase.startWithInput(taskId));
            checkPhaseStatus(currentPhase, changes);
        }

        return changes;
    }

    public Changes taskPreconditionValidated(String taskId) {
        Changes changes = new Changes();

        if (hasCurrentPhase()) {
            Phase currentPhase = getCurrentPhase();
            changes.addAll(currentPhase.taskPreconditionValidated(taskId));
            checkPhaseStatus(currentPhase, changes);
        }

        return changes;
    }

    public Changes failTask(String taskId, String failReason) {
        return failTask(taskId, failReason, User.AUTHENTICATED_USER, false);
    }

    public Changes failTask(String taskId, String failReason, boolean fromAbort) {
        return failTask(taskId, failReason, User.AUTHENTICATED_USER, fromAbort);
    }

    public Changes failTask(String taskId, String failReason, User user, boolean fromAbort) {
        Changes changes = new Changes();
        Task task = getTask(taskId);

        for (Phase phase : phases) {
            Changes updatedPhaseItems = phase.failTask(taskId, failReason, user, fromAbort);

            if (updatedPhaseItems.hasUpdatedItems()) {
                changes.addAll(updatedPhaseItems);
                if (!task.isTaskFailureHandlerEnabled() && isAbortOnFailure()) {
                    String lastCommentOnTask = getLastComment(task);
                    changes.addAll(abort(lastCommentOnTask.isEmpty() ?
                            format("Task '%s' failed", task.getTitle()) :
                            format("Task '%s' failed because of: '%s'", task.getTitle(), lastCommentOnTask)));
                } else if (phase.isFailed()) {
                    changes.addAll(fail());
                } else if (phase.isFailing()) {
                    changes.addAll(failing());
                }
            }
        }

        return changes;
    }

    private String getLastComment(Task task) {
        if (task.getComments().size() != 0) {
            return task.getComments().get(task.getComments().size() - 1).getText();
        }
        return "";
    }

    private Changes fail() {
        Changes changes = new Changes();
        setStatus(FAILED);
        changes.update(this);
        changes.addOperation(new ReleaseFailOperation(this));
        return changes;
    }

    private Changes failing() {
        Changes changes = new Changes();
        setStatus(FAILING);
        changes.update(this);
        changes.addOperation(new ReleaseStartFailingOperation(this));
        return changes;
    }

    public Changes abort(String abortComment) {
        return abort(false, abortComment);
    }

    public Changes abortAsPartOfBulkOperation(String abortComment) {
        return abort(true, abortComment);
    }

    public Changes retryTask(String taskId) {
        Changes changes = new Changes();

        for (Phase phase : phases) {
            Changes updatedPhaseItems = phase.retryTask(taskId);

            if (updatedPhaseItems.hasUpdatedItems()) {
                setStatus(IN_PROGRESS);
                changes.addAll(updatedPhaseItems);
                changes.addOperation(new ReleaseRetryOperation(this));
                checkPhaseStatus(phase, changes);
                changes.update(this);
            }
        }

        return changes;
    }

    public Phase getNextPhase(Phase currentPhase) {
        int currentPhasePosition = phases.indexOf(currentPhase) + 1;
        ListIterator<Phase> iterator = phases.listIterator(currentPhasePosition);
        return iterator.hasNext() ? iterator.next() : null;
    }

    public boolean hasNextPhase(Phase phase) {
        return getNextPhase(phase) != null;
    }

    public void addPhase(Phase phase) {
        addPhase(phase, phases.size());
    }

    public void addPhase(Phase phase, int position) {
        phases.add(position, phase);
        phase.setRelease(this);
    }

    public void deletePhase(Phase phase) {
        phases.remove(phase);
    }

    public List<Task> getAllTasks() {
        List<Task> tasks = new ArrayList<>();

        if (phases != null) {
            for (Phase phase : phases) {
                tasks.addAll(phase.getAllTasks());
            }
        }

        return tasks;
    }

    public Task getTask(String taskId) {
        if (taskId != null) {
            String folderlessTaskId = Ids.getFolderlessId(taskId);
            for (Task task : getAllTasks()) {
                if (task.getId() != null && task.getId().endsWith(folderlessTaskId)) {
                    return task;
                }
            }
        }
        return null;
    }

    public Attachment getAttachment(String attachmentId) {
        for (Attachment attachment : attachments) {
            if (attachmentId.equals(attachment.getId())) {
                return attachment;
            }
        }
        return null;
    }

    public Phase movePhase(Integer originIndex, Integer targetIndex) {
        Phase phaseToMove = phases.get(originIndex);
        Phase targetPhase = phases.get(targetIndex);
        checkArgument(phaseToMove.isPlanned() && targetPhase.isPlanned(), "Only planned phases can be moved.");
        phases.remove(phaseToMove);
        phases.add(targetIndex, phaseToMove);
        return phaseToMove;
    }

    public Phase getPhase(Integer index) {
        return phases.get(index);
    }

    public Phase getPhase(final String phaseId) {
        return getPhases().stream()
                .filter(phase -> Objects.equals(Ids.getName(phase.getId()), Ids.getName(phaseId)))
                .findAny()
                .orElse(null);
    }

    public boolean hasPhase(String phaseId) {
        return getPhase(phaseId) != null;
    }

    public void addTeam(Team team) {
        checkArgument(!hasTeam(team.getTeamName()), "Release '%s' already has a team named %s", getTitle(), team.getTeamName());
        teams.add(team);
    }

    public void deleteTeam(String teamId) {
        for (Iterator<Team> i = teams.iterator(); i.hasNext(); ) {
            Team team = i.next();
            if (team.getId().equals(teamId)) {
                if (isInRootFolder(getId())) {
                    checkArgument(!team.isSystemTeam(), "The predefined '%s' team can not be deleted.", team.getTeamName());
                }
                i.remove();
            }
        }
    }

    public void addBelow(String phaseId, Phase addedPhase) {
        addedPhase.setRelease(this);
        addBelow(phaseId, singletonList(addedPhase));
    }

    void addBelow(String phaseId, List<Phase> phasesToAdd) {
        for (int i = 0; i < getPhases().size(); i++) {
            Phase phase = getPhase(i);
            if (phase.getId().equals(phaseId)) {
                phases.addAll(i + 1, phasesToAdd);
                phasesToAdd.forEach(added -> added.setRelease(this));
            }
        }
    }

    public Set<VariableReference> collectVariableReferences() {
        return collectVariableReferences(this);
    }

    public Set<VariableReference> collectVariableReferences(VisitableItem updated) {
        return VariableCollectingVisitor.collectFrom(updated);
    }

    public List<GateTask> getAllGates() {
        return getAllTasksOfType(GateTask.class);
    }

    public List<UserInputTask> getAllUserInputTasks() {
        return getAllTasksOfType(UserInputTask.class);
    }

    public <T> List<T> getAllTasksOfType(Class<T> clazz) {
        //noinspection unchecked
        return getAllTasks().stream()
                .filter(clazz::isInstance)
                .map(task -> (T) task)
                .collect(toList());
    }

    public Date findFirstSetDate() {
        for (Phase phase : this.getPhases()) {
            if (phase.hasScheduledStartDate()) {
                return phase.getScheduledStartDate();
            } else {
                for (Task task : phase.getAllTasks()) {
                    if (task.hasScheduledStartDate()) {
                        return task.getScheduledStartDate();
                    }
                }
            }
        }
        return DateTime.now().withHourOfDay(9).withMinuteOfHour(0).withSecondOfMinute(0).withMillisOfSecond(0).toDate();
    }

    public void clearComments() {
        this.getPhases().forEach((phase) -> phase.getAllTasks().forEach(Task::clearComments));
    }

    public boolean hasNoAutomatedTaskRunning() {
        return getAllTasks().stream().noneMatch(IS_AUTOMATED_AND_IN_PROGRESS);
    }

    public boolean isDone() {
        return status == COMPLETED;
    }

    public boolean isPlannedOrActive() {
        return isPlanned() || isActive();
    }

    public boolean isActive() {
        return status != null && status.isActive();
    }

    public boolean isDefunct() {
        return isAborted() || isDone();
    }

    public boolean isUpdatable() {
        return !isDefunct();
    }

    public boolean isAborted() {
        return status == ABORTED;
    }

    public boolean isFailing() {
        return status == FAILING;
    }

    public boolean isFailed() {
        return status == FAILED;
    }

    public boolean isPaused() {
        return status == PAUSED;
    }

    public boolean isTemplate() {
        return status == TEMPLATE;
    }

    public boolean isPlanned() {
        return PLANNED == status;
    }

    public boolean isInProgress() {
        return status == IN_PROGRESS;
    }

    public boolean isTutorial() {
        return tutorial;
    }

    public void setTutorial(boolean tutorial) {
        this.tutorial = tutorial;
    }

    @PublicApiMember
    public boolean isAbortOnFailure() {
        return abortOnFailure;
    }

    @PublicApiMember
    public void setAbortOnFailure(boolean abortOnFailure) {
        this.abortOnFailure = abortOnFailure;
    }

    @PublicApiMember
    public boolean isArchiveRelease() {
        return archiveRelease;
    }

    public void setArchiveRelease(boolean archiveRelease) {
        this.archiveRelease = archiveRelease;
    }

    @PublicApiMember
    public boolean isAllowPasswordsInAllFields() {
        return allowPasswordsInAllFields;
    }

    @PublicApiMember
    public void setAllowPasswordsInAllFields(final boolean allowPasswordsInAllFields) {
        this.allowPasswordsInAllFields = allowPasswordsInAllFields;
    }

    @PublicApiMember
    public boolean isDisableNotifications() {
        return disableNotifications;
    }

    @PublicApiMember
    public void setDisableNotifications(boolean disableNotifications) {
        this.disableNotifications = disableNotifications;
    }

    @PublicApiMember
    public boolean isAllowConcurrentReleasesFromTrigger() {
        return allowConcurrentReleasesFromTrigger;
    }

    @PublicApiMember
    public void setAllowConcurrentReleasesFromTrigger(boolean allowConcurrentReleasesFromTrigger) {
        this.allowConcurrentReleasesFromTrigger = allowConcurrentReleasesFromTrigger;
    }

    @PublicApiMember
    public String getOriginTemplateId() {
        return originTemplateId;
    }

    public void setOriginTemplateId(String originTemplateId) {
        this.originTemplateId = originTemplateId;
    }

    @PublicApiMember
    public boolean isCreatedFromTrigger() {
        return createdFromTrigger;
    }

    public void setCreatedFromTrigger(boolean createdFromTrigger) {
        this.createdFromTrigger = createdFromTrigger;
    }

    public Release getRelease() {
        return this;
    }

    @Override
    public Integer getReleaseUid() {
        return getCiUid();
    }

    @Override
    public void setReleaseUid(Integer releaseUid) {
        if (ciUid == null && releaseUid != null) {
            this.ciUid = releaseUid;
        }
    }

    public String getDisplayPath() {
        return getTitle();
    }

    public Team getTeamWithId(String id) {
        for (Team team : teams) {
            if (team.getId().equals(id)) {
                return team;
            }
        }

        throw new IllegalStateException("Release " + this.getId() + " doesn't contain team with id " + id);
    }

    public List<PlanItem> getChildren() {
        List<PlanItem> planItems = new ArrayList<>();
        planItems.addAll(phases);
        planItems.addAll(getAllTasks());
        return planItems;
    }

    public List<PlanItem> getAllPlanItems() {
        List<PlanItem> planItems = new ArrayList<>();
        planItems.add(this);
        planItems.addAll(getChildren());
        return planItems;
    }

    private void freezeVariables() {
        Map<String, ValueWithInterpolation> variables = getAllVariableValuesAsStringsWithInterpolationInfo();
        if (variables == null || variables.isEmpty()) return;

        setDescription(replaceAllWithInterpolation(getDescription(), variables, new HashSet<>(), false));
    }

    @Override
    public String toString() {
        return format("Release {id=%s, title=%s, status=%s}", this.getId(), this.getTitle(), this.getStatus());
    }

    public void checkDatesValidityForRelease() {
        checkArgument(dueDate != null, "Due date must be set");
        checkDatesValidityForTemplate();
    }

    public void checkDatesValidityForTemplate() {
        checkArgument(scheduledStartDate != null, "Scheduled start date must be set");
        checkDatesValidity();
    }

    public void checkDatesValidity() {
        checkDatesValidity(scheduledStartDate, dueDate, plannedDuration);
    }

    public void updateDatesForRelease(Date scheduledStartDate, Date dueDate, Integer plannedDuration) {
        checkArgument(dueDate != null, "Due date must be set");
        updateDatesForTemplate(scheduledStartDate, dueDate, plannedDuration);
    }

    public void updateDatesForTemplate(Date scheduledStartDate, Date dueDate, Integer plannedDuration) {
        checkArgument(scheduledStartDate != null, "Scheduled start date must be set");
        updateDates(scheduledStartDate, dueDate, plannedDuration);
    }

    public Date getQueryableStartDate() {
        return queryableStartDate;
    }

    public void setQueryableStartDate(Date queryableStartDate) {
        this.queryableStartDate = queryableStartDate;
    }

    public Date getQueryableEndDate() {
        return queryableEndDate;
    }

    public void setQueryableEndDate(Date queryableEndDate) {
        this.queryableEndDate = queryableEndDate;
    }

    @PublicApiMember
    public String getScriptUsername() {
        return scriptUsername;
    }

    @PublicApiMember
    public void setScriptUsername(String scriptUsername) {
        this.scriptUsername = scriptUsername;
    }

    @PublicApiMember
    public String getScriptUserPassword() {
        return scriptUserPassword;
    }

    @PublicApiMember
    public void setScriptUserPassword(String scriptUserPassword) {
        this.scriptUserPassword = scriptUserPassword;
    }

    @PublicApiMember
    public String getUrl() {
        ServerUrl serverUrl = (ServerUrl) $metadata.get("serverUrl");
        if (serverUrl == null) {
            return null;
        }
        return ServerUrl.buildUrl(serverUrl.url(), this);
    }

    @PublicApiMember
    public Map<String, String> getVariableMapping() {
        return variableMapping;
    }

    @PublicApiMember
    public void setVariableMapping(Map<String, String> variableMapping) {
        this.variableMapping = variableMapping;
    }

    @PublicApiMember
    public RiskProfile getRiskProfile() {
        return riskProfile;
    }

    @PublicApiMember
    public void setRiskProfile(final RiskProfile riskProfile) {
        this.riskProfile = riskProfile;
    }

    public String getStartedFromTaskId() {
        return startedFromTaskId;
    }

    public void setStartedFromTaskId(final String startedFromTaskId) {
        this.startedFromTaskId = startedFromTaskId;
    }

    @Override
    public void setStartDate(Date value) {
        super.setStartDate(value);
        this.queryableStartDate = getStartOrScheduledDate();
    }

    @Override
    public void setScheduledStartDate(Date scheduledStartDate) {
        super.setScheduledStartDate(scheduledStartDate);
        this.queryableStartDate = getStartOrScheduledDate();
    }

    @Override
    public void setEndDate(Date value) {
        super.setEndDate(value);
        this.queryableEndDate = getEndOrDueDate();
    }

    @Override
    public void setDueDate(Date value) {
        super.setDueDate(value);
        this.queryableEndDate = getEndOrDueDate();
    }

    public void updateTeam(final Team newTeam) {
        Optional<Team> optionalTeam = teams.stream().filter(team -> team.getId().equals(newTeam.getId())).findAny();
        if (optionalTeam.isPresent()) {
            Team team = optionalTeam.get();

            if (team.isSystemTeam()) {
                checkArgument(newTeam.getTeamName().equals(team.getTeamName()), "Cannot rename a system team");
            }

            team.setTeamName(newTeam.getTeamName());
            team.setRoles(newTeam.getRoles());
            team.setMembers(newTeam.getMembers());
            team.setPermissions(newTeam.getPermissions());
        } else {
            logger.warn("Tried to update an unknown team {}", newTeam.getId());
        }
    }

    public void updateRealFlagStatus() {
        FlagStatus tasksFlagStatus;
        if (getAllTasks().size() > 0) {
            Task worstTask = Collections.max(getAllTasks(),
                    Comparator.comparingInt(t -> t.getFlagStatus().getRisk()));
            tasksFlagStatus = worstTask.getFlagStatus();
        } else {
            tasksFlagStatus = FlagStatus.OK;
        }
        realFlagStatus = tasksFlagStatus.getRisk() > flagStatus.getRisk() ? tasksFlagStatus : flagStatus;
    }

    public boolean hasScriptUsername() {
        return scriptUsername != null;
    }

    public List<Task> getActiveTasks() {
        return getAllTasks().stream()
                .filter(Task::isActive)
                .collect(toList());
    }

    public boolean isArchived() {
        return archived;
    }

    public void setArchived(boolean isArchived) {
        this.archived = isArchived;
    }

    public void accept(ReleaseVisitor visitor) {
        visitor.visit(this);
        for (Phase phase : phases) {
            phase.accept(visitor);
        }
        for (ReleaseExtension extension : extensions) {
            extension.accept(visitor);
        }
        for (Variable variable : variables) {
            variable.accept(visitor);
        }
    }

    @Override
    public List<UsagePoint> getVariableUsages() {
        return asList(
                new PropertyUsagePoint(this, "description"),
                new PropertyUsagePoint(this, "scriptUsername"),
                new PropertyUsagePoint(this, SCRIPT_USER_PASSWORD_VARIABLE_MAPPING_KEY),
                new ReleaseVariablesUsagePoint(this)
        );
    }

    @Override
    public Map<String, InternalMetadata> get$metadata() {
        return $metadata;
    }

    public String findFolderId() {
        return Ids.findFolderId(getId());
    }

    public boolean hasTeam(String teamName) {
        return getTeams().stream().anyMatch(team -> Strings.nullToEmpty(team.getTeamName()).equalsIgnoreCase(teamName));
    }

    public Set<String> getTeamsOf(String username, List<Role> userRoles) {
        return getTeams().stream()
                .filter(team -> team.hasMember(username) || team.hasAnyRole(userRoles))
                .map(Team::getTeamName)
                .collect(Collectors.toSet());
    }

    public Set<String> getPermissions(String user, List<Role> userRoles) {
        return getPermissions(singletonList(user), userRoles);
    }

    public Set<String> getPermissions(Collection<String> principals, List<Role> userRoles) {
        return getTeams().stream()
                .filter(team -> team.hasAnyMember(principals) || team.hasAnyRole(userRoles))
                .flatMap(team -> team.getPermissions().stream())
                .collect(Collectors.toSet());
    }

    public boolean isAutoStart() {
        return autoStart;
    }

    public void setAutoStart(final boolean autoStart) {
        this.autoStart = autoStart;
    }

    public boolean canAutomaticallyStartReleaseNow() {
        return canScheduleReleaseStart() && new Date().after(getScheduledStartDate());
    }

    public boolean canScheduleReleaseStart() {
        return status == PLANNED && autoStart && getScheduledStartDate() != null;
    }

    public boolean isPending() {
        return isPlanned() && isAutoStart();
    }

    public void deleteTask(Task task) {
        getPhases().forEach(phase -> phase.deleteTask(task));
    }

    public void replaceTask(Task task) {
        getPhases().forEach(phase -> phase.replaceTask(task));
    }

    public List<Phase> getPhasesByTitle(final String phaseTitle) {
        return getPhases().stream()
                .filter(phase -> phase.hasTitle(phaseTitle))
                .collect(Collectors.toList());
    }

    public List<Phase> getPhasesContainingInTitle(final String partialTitleToLookFor) {
        return filterPhasesContainingInTitle(getPhases(), partialTitleToLookFor);
    }

    public List<Phase> filterPhasesContainingInTitle(List<Phase> phases, final String partialTitleToLookFor) {
        return phases.stream()
                .filter(phase -> phase.isTitleContaining(partialTitleToLookFor))
                .collect(Collectors.toList());
    }

    public List<Task> getTasksByTitle(final String phaseTitle, final String taskTitle) {
        return getAllTasks().stream()
                .filter(task -> task.hasTitle(taskTitle) &&
                        (isNullOrEmpty(phaseTitle) || task.getPhase().hasTitle(phaseTitle)))
                .collect(Collectors.toList());
    }

    public Changes restorePhases(List<Phase> phases) {
        Changes changes = new Changes();
        addBelow(getCurrentPhase().getId(), phases);
        setStatus(PAUSED);
        changes.update(this);
        changes.addOperation(new ReleasePauseOperation(this));
        return changes;
    }

    public void fixId(String folderId) {
        rewriteWithNewId(this, formatWithFolderId(folderId, getId()));
    }

    private Changes start(boolean isPartOfBulkOperation, boolean releaseStartedImmediatelyAfterBeingCreated) {
        checkState(!this.hasBeenStarted(), "Only not started releases may be started. Current release status is: %s", this.status);

        Changes changes = new Changes();

        this.setStatus(IN_PROGRESS);

        if (!releaseStartedImmediatelyAfterBeingCreated) {
            Date currentDate = new Date();
            this.setStartDate(currentDate);
            if (plannedDuration != null) {
                DateTime newDueDate = new DateTime(currentDate).plusSeconds(plannedDuration);
                this.setDueDate(newDueDate.toDate());
            }

        } else {
            this.setStartDate(scheduledStartDate);
        }

        changes.update(this);
        changes.addOperation(new ReleaseStartOperation(this, isPartOfBulkOperation));

        if (hasPhase()) {
            Phase firstPhase = this.getPhases().get(0);
            startActivablePhase(firstPhase, changes);
        } else {
            complete(changes);
        }

        return changes;
    }

    private Changes abort(boolean isPartOfBulkOperation, final String abortComment) {
        checkState(getStatus() != TEMPLATE, "Can't abort a template release");
        checkState(!Strings.isNullOrEmpty(abortComment), "Can't abort a release without comment");

        Changes changes = new Changes();

        setAbortComment(abortComment);
        setStatus(ABORTED);
        setEndDate(new Date());
        changes.update(this);
        changes.addOperation(new ReleaseAbortOperation(this, isPartOfBulkOperation));
        changes.addOperation(new ReleaseAbortScriptsExecution(this));
        phases.stream()
                .filter(phase -> !phase.isDone())
                .forEach(phase -> changes.addAll(phase.abort()));

        return changes;
    }
}
