package com.xebialabs.xlrelease.domain.delivery;

import java.util.Collections;
import java.util.Date;
import java.util.List;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.repository.Ids;

import static com.xebialabs.xlrelease.domain.Task.CATEGORY_INPUT;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

@Metadata(versioned = false, virtual = true, description = "Condition for a transition to execute automatically.")
@PublicApiRef
@ShowOnlyPublicApiMembers
public abstract class Condition extends BaseConfigurationItem {
    @Property(label = "Is satisfied", description = "Is the transition condition satisfied by an internal or external event.")
    private boolean satisfied = false;

    @Property(label = "Satisfied date", description = "When was the condition satisfied.")
    private Date satisfiedDate;

    // --

    public abstract String getDescription();

    public void validate(Delivery delivery) {
        // NOOP
    }

    public boolean isActive() {
        return !isSatisfied();
    }

    public Condition withId(String id) {
        this.id = id;
        return this;
    }
    public List<PropertyDescriptor> getInputProperties() {
        return this.getType().getDescriptor().getPropertyDescriptors()
                .stream()
                .filter(pd -> CATEGORY_INPUT.equals(pd.getCategory()))
                .collect(toList());
    }

    public List<Condition> markAsSatisfied(String conditionId, Date satisfiedDate) {
        if (isActive() && Ids.getName(conditionId).equals(Ids.getName(getId()))) {
            this.satisfied = true;
            this.satisfiedDate = satisfiedDate;
            return singletonList(this);
        }

        return Collections.emptyList();
    }

    public boolean isLeaf() {
        return true;
    }

    public List<Condition> getAllConditions() {
        return singletonList(this);
    }

    public List<Condition> getLeafConditions() {
        return singletonList(this);
    }

    public void reset() {
        setSatisfied(false);
        setSatisfiedDate(null);
    }
    // --

    @PublicApiMember
    public boolean isSatisfied() {
        return satisfied;
    }

    public void setSatisfied(final boolean satisfied) {
        this.satisfied = satisfied;
    }

    @PublicApiMember
    public Date getSatisfiedDate() {
        return satisfiedDate;
    }

    public void setSatisfiedDate(final Date satisfiedDate) {
        this.satisfiedDate = satisfiedDate;
    }
}
