package com.xebialabs.xlrelease.domain.delivery.conditions;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.delivery.Condition;
import com.xebialabs.xlrelease.domain.delivery.Delivery;

import static com.xebialabs.xlrelease.domain.Task.CATEGORY_INPUT;
import static com.xebialabs.xlrelease.domain.delivery.conditions.ConditionGroup.Operator.AND;
import static com.xebialabs.xlrelease.domain.delivery.conditions.ConditionGroup.Operator.OR;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

@Metadata(versioned = false, description = "Composite condition comprised of multiple conditions joined by an AND or OR operator.")
@PublicApiRef
@ShowOnlyPublicApiMembers
public class ConditionGroup extends Condition {
    @Property(category = CATEGORY_INPUT, description = "Operator to apply when merging the child conditions.")
    private ConditionGroup.Operator operator;

    @Property(asContainment = true, description = "Nested conditions.")
    private List<Condition> conditions = new ArrayList<>();

    // --

    @Override
    public List<Condition> markAsSatisfied(String conditionId, Date satisfiedDate) {
        List<Condition> changes = conditions.stream()
                .flatMap(condition -> condition.markAsSatisfied(conditionId, satisfiedDate).stream())
                .collect(toList());
        if (isActive()) {
            if (operator == OR && conditions.stream().anyMatch(Condition::isSatisfied) ||
                    operator == AND && conditions.stream().allMatch(Condition::isSatisfied)) {
                setSatisfied(true);
                setSatisfiedDate(satisfiedDate);
                changes.add(this);
            }
        }
        return changes;
    }

    @Override
    public String getDescription() {
        return conditions.stream().map(Condition::getDescription).collect(joining(" " + operator.getValue() + " "));
    }

    @Override
    public void validate(Delivery delivery) {
        conditions.forEach(condition -> condition.validate(delivery));
    }

    public void addCondition(final Condition condition) {
        if (conditions == null) {
            conditions = new ArrayList<>();
        }
        conditions.add(condition);
    }

    public ConditionGroup withOperator(final Operator operator) {
        this.operator = operator;
        return this;
    }

    public ConditionGroup withCondition(final Condition condition) {
        addCondition(condition);
        return this;
    }

    public boolean isLeaf() {
        return false;
    }

    @Override
    public List<Condition> getAllConditions() {
        List<Condition> allConditions = new ArrayList<>(super.getAllConditions());
        conditions.forEach(child -> allConditions.addAll(child.getAllConditions()));
        return allConditions;
    }

    @Override
    public List<Condition> getLeafConditions() {
        return conditions.stream().flatMap(condition -> condition.getLeafConditions().stream()).collect(toList());
    }

    // --

    public ConditionGroup.Operator getOperator() {
        return operator;
    }

    public void setOperator(final ConditionGroup.Operator operator) {
        this.operator = operator;
    }

    public List<Condition> getConditions() {
        return conditions;
    }

    public void setConditions(final List<Condition> conditions) {
        this.conditions = conditions;
    }

    // --

    public enum Operator {
        AND("AND"),
        OR("OR");

        private String value;

        Operator(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }
}
