package com.xebialabs.xlrelease.domain.variables;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.BaseConfiguration;
import com.xebialabs.xlrelease.domain.ReleaseVisitor;
import com.xebialabs.xlrelease.domain.VisitableItem;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.xebialabs.xlrelease.variable.VariableHelper.isCiPropertyVariable;
import static com.xebialabs.xlrelease.variable.VariableHelper.isFolderVariable;
import static com.xebialabs.xlrelease.variable.VariableHelper.isGlobalVariable;
import static java.lang.String.format;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(virtual = true, versioned = false)
public abstract class Variable extends BaseConfiguration implements Cloneable, VisitableItem {

    @PublicApiMember
    @Property(description = "The unique name of the variable in the way it is used in template or release, without curly braces")
    protected String key;

    @PublicApiMember
    @Property(required = false, defaultValue = "true", description = "Shows if an empty value is a valid value for this variable")
    protected boolean requiresValue;

    @PublicApiMember
    @Property(required = false, defaultValue = "true", description = "Shows if this variable will be shown on create release page")
    protected boolean showOnReleaseStart;

    @PublicApiMember
    @Property(required = false, description = "Label of the variable")
    protected String label;

    @PublicApiMember
    @Property(required = false, description = "Description of the variable")
    protected String description;

    @PublicApiMember
    @Property(required = false, description = "Configuration of the variable values provider")
    protected ValueProviderConfiguration valueProvider;

    @Property(required = false, description = "Determines whether the variable value is inherited from the template variable. Used in the Create Release Task only.")
    protected boolean inherited;

    /**
     * Gets underlying value of this variable, or default value for templates. The specific type of value depends on the type of variable.
     *
     * @return the underlying value of this variable, or default value for templates.
     */
    @PublicApiMember
    // This annotation is needed for abstract member "value" to appear in REST API
    public abstract Object getValue();

    /**
     * Returns empty value for the specific sub type. Empty value is used when replacing unset optional variables.
     *
     * @return empty value for the specific sub type.
     */
    public abstract Object getEmptyValue();

    /**
     * Test if value is unset or empty. Rely on the {@link #getEmptyValue()} implementation.
     *
     * @return true if value is null or empty.
     */
    @PublicApiMember
    public boolean isValueEmpty() {
        Object variableValue = getValue();
        return variableValue == null || variableValue.equals(getEmptyValue());
    }

    /**
     * Sets the underlying value of this variable from given object. If a conversion happens to be needed
     * and fails, then an {@link IllegalArgumentException} will be thrown.
     *
     * @param newValue the value to set: either of the type needed by this variable type, or one of convertible types.
     * @throws IllegalArgumentException if conversion of the value fails.
     */
    public abstract void setUntypedValue(Object newValue) throws IllegalArgumentException;

    @PublicApiMember
    public String getKey() {
        return key;
    }

    @PublicApiMember
    public void setKey(final String key) {
        this.key = key;
    }

    @PublicApiMember
    public boolean getRequiresValue() {
        return requiresValue;
    }

    @PublicApiMember
    public void setRequiresValue(boolean requiresValue) {
        this.requiresValue = requiresValue;
    }

    @PublicApiMember
    public boolean getShowOnReleaseStart() {
        return showOnReleaseStart;
    }

    @PublicApiMember
    public void setShowOnReleaseStart(boolean showOnReleaseStart) {
        this.showOnReleaseStart = showOnReleaseStart;
    }

    @PublicApiMember
    public String getLabel() {
        return label;
    }

    @PublicApiMember
    public void setLabel(final String label) {
        this.label = label;
    }

    @PublicApiMember
    public String getDescription() {
        return description;
    }

    @PublicApiMember
    public void setDescription(final String description) {
        this.description = description;
    }

    public void checkValidity() {
        checkNotNull(getKey());
        checkArgument(!isCiPropertyVariable(getKey()), format("You cannot create a variable with a key like release.* (%s)", getKey()));
        checkArgument(!isGlobalVariable(getKey()), format("You cannot create a variable with a key like global.* (%s)", getKey()));
        checkArgument(!isFolderVariable(getKey()), format("You cannot create a variable with a key like folder.* (%s)", getKey()));
    }

    public void checkGlobalVariableValidity() {
        checkNotNull(getKey());
        checkArgument(isGlobalVariable(getKey()),
                "Global variable names must start with \"global.\", but [%s] was supplied", getKey());
        checkArgument(!requiresValue && !showOnReleaseStart,
                "Global variables must have requiresValue and showOnReleaseStart flags unchecked");
    }

    public void checkFolderVariableValidity() {
        checkNotNull(getKey());
        checkArgument(isFolderVariable(getKey()),
                "Folder variable names must start with \"folder.\", but [%s] was supplied", getKey());
        checkArgument(!requiresValue && !showOnReleaseStart,
                "Folder variables must have requiresValue and showOnReleaseStart flags unchecked");
    }

    // We cannot make Variable parameterizable as UDM type system does not support it
    public static abstract class VariableWithValue<V> extends Variable {

        @Override
        public abstract V getValue();

        public abstract void setValue(V value);

    }

    public abstract boolean isPassword();

    public abstract String getValueAsString();

    public abstract String getEmptyValueAsString();

    public abstract boolean isValueAssignableFrom(Object value);

    protected String escapeQuotes(String value) {
        return value.replace("'", "\\'");
    }

    public ValueProviderConfiguration getValueProvider() {
        return valueProvider;
    }

    public void setValueProvider(final ValueProviderConfiguration valueProvider) {
        this.valueProvider = valueProvider;
        if (valueProvider != null) {
            valueProvider.setVariable(this);
        }
    }

    @Override
    public void accept(ReleaseVisitor visitor) {
        if (getValueProvider() != null) {
            visitor.visit(getValueProvider());
        }
    }

    public boolean isInherited() {
        return inherited;
    }

    public void setInherited(boolean inherited) {
        this.inherited = inherited;
    }

    public Variable clone() throws CloneNotSupportedException {
        return (Variable) super.clone();
    }
}
