package com.xebialabs.xlrelease.domain.variables.reference;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.repository.CiProperty;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.xlrelease.utils.Collectors.toMap;
import static java.util.function.Function.identity;

/**
 * This usage point means that a variable is contained as a value in one of entries
 * of the <code>variableMapping</code>.
 */
public class VariableMappingUsagePoint implements UsagePoint {
    private final CiProperty targetProperty;
    private ConfigurationItem ci;
    private String fqPropertyName;

    public VariableMappingUsagePoint(ConfigurationItem ci, String fqPropertyName, CiProperty targetProperty) {
        this.ci = ci;
        this.fqPropertyName = fqPropertyName;
        this.targetProperty = targetProperty;
    }

    @Override
    public Map<String, VariableReference.VariableUsageType> collectVariables() {
        final PropertyUsagePoint propertyUsagePoint = new PropertyUsagePoint(targetProperty);
        return VariableHelper.collectVariables(getVariableMapping(ci).get(fqPropertyName))
                .stream()
                .collect(toMap(identity(), propertyUsagePoint::getVariableUsageType));
    }

    @Override
    public Set<ConfigurationItem> replaceVariable(final Variable variable, final VariableOrValue replacement) {
        Set<ConfigurationItem> updatedItems = newHashSet(ci);
        Map<String, String> variableMapping = getVariableMapping(ci);

        if (replacement.getVariable() != null) {
            variableMapping.put(fqPropertyName, VariableHelper.formatVariableIfNeeded(replacement.getVariable()));
        } else {
            targetProperty.setValue(replacement.getValue());
            updatedItems.add(targetProperty.getParentCi());
            variableMapping.remove(fqPropertyName);
        }

        return updatedItems;
    }

    private Map<String, String> getVariableMapping(ConfigurationItem ci) {
        if (ci.hasProperty("variableMapping")) {
            return ci.getProperty("variableMapping");
        } else {
            return new HashMap<>();
        }
    }

    @Override
    public CiProperty getTargetProperty() {
        return targetProperty;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        final VariableMappingUsagePoint that = (VariableMappingUsagePoint) o;
        return ci.getType().equals(ci.getType()) && ci.equals(that.ci) && fqPropertyName.equals(that.fqPropertyName);
    }

    @Override
    public int hashCode() {
        int result = ci.hashCode();
        result = 31 * result + fqPropertyName.hashCode();
        return result;
    }
}
