package com.xebialabs.xlrelease.events

import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.variables.Variable

import java.util.{List => JList}

/**
 * A bundle of an "interceptable action" and  an "event" that need to be posted
 * when corresponding operation is performed. The "action" should be posted
 * before any changes are persisted, and the "event" - after changes were successfully saved.
 */
trait XLReleaseOperation {
  def action: InterceptableAction

  def event: XLReleaseEvent
}

// Release operations
trait ReleaseOperation extends XLReleaseOperation

case class ReleaseStartOperation(release: Release, isPartOfBulkOperation: Boolean) extends ReleaseOperation {
  val action = ReleaseStartingAction(release.getId)
  val event = ReleaseStartedEvent(release, isPartOfBulkOperation)

  def this(release: Release) = this(release, false)
}

case class ReleaseAbortOperation(release: Release, isPartOfBulkOperation: Boolean) extends ReleaseOperation {
  val action = ReleaseAbortingAction(release.getId)
  val event = ReleaseAbortedEvent(release, isPartOfBulkOperation)

  def this(release: Release) = this(release, false)
}

case class ReleaseCompleteOperation(release: Release)
  extends ReleaseOperation {
  val action = ReleaseCompletingAction(release.getId)
  val event = ReleaseCompletedEvent(release)
}

case class ReleasePauseOperation(release: Release)
  extends ReleaseOperation {
  val action = ReleasePausingAction(release.getId)
  val event = ReleasePausedEvent(release)
}

case class ReleaseResumeOperation(release: Release)
  extends ReleaseOperation {
  val action = ReleaseResumingAction(release.getId)
  val event = ReleaseResumedEvent(release)
}

case class ReleaseFailOperation(release: Release)
  extends ReleaseOperation {
  val action = null
  val event = ReleaseFailedEvent(release)
}

case class ReleaseStartFailingOperation(release: Release)
  extends ReleaseOperation {
  val action = null
  val event = ReleaseStartedFailingEvent(release)
}

case class ReleaseRetryOperation(release: Release)
  extends ReleaseOperation {
  val action = ReleaseRetryingAction(release.getId)
  val event = ReleaseRetriedEvent(release)
}

case class ReleaseAbortScriptsExecution(release: Release)
  extends XLReleaseOperation {
  val action = ReleaseAbortScriptsAction(release.getId)
  val event = null
}

// Phase operations
case class PhaseStartOperation(phase: Phase)
  extends XLReleaseOperation {
  val action = PhaseStartingAction(phase.getId)
  val event = PhaseStartedEvent(phase)
}

case class PhaseFailOperation(phase: Phase)
  extends XLReleaseOperation {
  val action = null
  val event = PhaseFailedEvent(phase)
}

case class PhaseStartFailingOperation(phase: Phase)
  extends XLReleaseOperation {
  val action = null
  val event = PhaseStartedFailingEvent(phase)
}

case class PhaseRetryOperation(phase: Phase)
  extends XLReleaseOperation {
  val action = PhaseRetryingAction(phase.getId)
  val event = PhaseRetriedEvent(phase)
}

case class PhaseCompleteOperation(phase: Phase)
  extends XLReleaseOperation {
  val action = PhaseCompletingAction(phase.getId)
  val event = PhaseCompletedEvent(phase)
}

final case class PhaseCloseOperation(phase: Phase)
  extends XLReleaseOperation {
  val action = PhaseClosingAction(phase.getId)
  val event = PhaseClosedEvent(phase)
}


// Task operations
case class TaskStartOperation(task: Task)
  extends TaskStartOrRetryOperation {
  val action = TaskStartingAction(task.getId)
  val event = TaskStartedEvent(task)
}

case class TaskWaitingForInputOperation(task: Task, unboundVariables: JList[String])
  extends XLReleaseOperation {
  val action = null
  val event = TaskWaitingForInputEvent(task, unboundVariables)
}

case class TaskCompleteOperation(task: Task, inAdvance: Boolean)
  extends XLReleaseOperation {
  val action = TaskCompletingAction(task.getId, inAdvance)
  val event = TaskCompletedEvent(task, inAdvance)
}

case class CompleteTaskJobExecutionOperation(task: Task, executionId: String)
  extends XLReleaseOperation {
  val action = null
  val event = TaskJobExecutedEvent(task.getId, executionId)
}

case class TaskSkipOperation(task: Task, inAdvance: Boolean)
  extends XLReleaseOperation {
  val action = TaskSkippingAction(task.getId, inAdvance)
  val event = TaskSkippedEvent(task, inAdvance)
}

case class TaskAbortOperation(task: Task)
  extends XLReleaseOperation {
  val action = TaskAbortingAction(task.getId)
  val event = TaskAbortedEvent(task)
}

case class TaskRetryOperation(task: Task)
  extends TaskStartOrRetryOperation {
  val action = TaskRetryingAction(task.getId)
  val event = TaskRetriedEvent(task)
}

case class TaskDelayOperation(task: Task)
  extends XLReleaseOperation {
  val action = TaskDelayingAction(task.getId)
  val event = TaskDelayedEvent(task)
}

case class TaskFailOperation(task: Task, reason: String)
  extends XLReleaseOperation {
  val action = TaskFailingAction(task.getId, reason)
  val event = TaskFailedEvent(task, reason)
}

case class TaskReopenOperation(task: Task)
  extends XLReleaseOperation {
  val action = TaskReopeningAction(task.getId)
  val event = TaskReopenedEvent(task)
}

case class TaskStartRecoveryOperation(task: Task)
  extends XLReleaseOperation {
  val action = null
  val event = TaskRecoveryStartedEvent(task)
}

case class TaskEndRecoveryOperation(task: Task)
  extends XLReleaseOperation {
  val action = null
  val event = TaskRecoveredEvent(task)
}

case class TaskStartAbortScriptOperation(task: Task)
  extends XLReleaseOperation {
  val action = null
  val event = TaskAbortScriptStartedEvent(task)
}

case class TaskCompleteAbortScriptOperation(task: Task)
  extends XLReleaseOperation {
  val action = null
  val event = TaskAbortScriptCompletedEvent(task)
}

case class TaskGroupFailingOperation(task: TaskGroup)
  extends XLReleaseOperation {
  val action = null
  val event = TaskGroupFailingEvent(task)
}

trait TaskStartOrRetryOperation extends XLReleaseOperation

// Variable operations
trait VariableCreateOperation extends XLReleaseOperation

trait VariableUpdateOperation extends XLReleaseOperation

trait VariableDeleteOperation extends XLReleaseOperation

case class ReleaseVariableCreateOperation(variable: Variable)
  extends VariableCreateOperation {
  val action = null
  val event = ReleaseVariableCreatedEvent(variable)
}

case class ReleaseVariableUpdateOperation(original: Variable, updated: Variable)
  extends VariableUpdateOperation {
  val action = null
  val event = ReleaseVariableUpdatedEvent(original, updated)
}

case class ReleaseVariablesUpdateOperation(original: JList[Variable], updated: JList[Variable], logDelete: Boolean)
  extends VariableUpdateOperation {
  val action = null
  val event = ReleaseVariablesUpdatedEvent(original, updated)
}

case class ReleaseVariableDeleteOperation(variable: Variable)
  extends VariableDeleteOperation {
  val action = null
  val event = ReleaseVariableDeletedEvent(variable)
}

case class GlobalVariableCreateOperation(variable: Variable, taskId: String)
  extends VariableCreateOperation {
  val action = null
  val event = GlobalVariableCreatedEvent(variable, ModifiedFromScriptTask(taskId))
}

case class GlobalVariableUpdateOperation(original: Variable, updated: Variable, taskId: String)
  extends VariableUpdateOperation {
  val action = null
  val event = GlobalVariableUpdatedEvent(original, updated, ModifiedFromScriptTask(taskId))
}

case class GlobalVariableDeleteOperation(variable: Variable, taskId: String)
  extends VariableDeleteOperation {
  val action = null
  val event = GlobalVariableDeletedEvent(variable, ModifiedFromScriptTask(taskId))
}

case class FolderVariableCreateOperation(variable: Variable, taskId: String)
  extends VariableCreateOperation {
  val action = null
  val event = FolderVariableCreatedEvent(variable, ModifiedFromScriptTask(taskId))
}

case class FolderVariableUpdateOperation(original: Variable, updated: Variable, taskId: String)
  extends VariableUpdateOperation {
  val action = null
  val event = FolderVariableUpdatedEvent(original, updated, ModifiedFromScriptTask(taskId))
}

case class FolderVariableDeleteOperation(variable: Variable, taskId: String)
  extends VariableDeleteOperation {
  val action = null
  val event = FolderVariableDeletedEvent(variable, ModifiedFromScriptTask(taskId))
}
